import logging
from datetime import date
from typing import Dict, Optional

from staff.budget_position.models.review import Review
from staff.lib.db import atomic
from staff.lib.sync_tools.rollupper import Rollupper

from staff.oebs.controllers.mappers import DataMapper
from staff.oebs.models import Review as OebsReview


logger = logging.getLogger(__name__)


class ReviewMapper(DataMapper):
    def __init__(self, object_dict: Dict):
        super().__init__(object_dict)
        self.mapping = (
            ('name', 'name'),
            ('description', 'description'),
            ('start_date', 'start_date'),
            ('end_date', 'end_date'),
            ('end_date', 'intranet_status', self._get_intranet_status),
            ('scheme_id', 'scheme_id'),
            ('target_bonus', 'target_bonus'),
            ('grant_type', 'grant_type'),
            ('grant_type_description', 'grant_type_description'),
        )

    @staticmethod
    def _get_intranet_status(end_date: Optional[date]) -> int:
        if end_date is None:
            return 1

        if end_date >= date.today():
            return 1

        return 0


class ReviewRollupper(Rollupper):
    model = OebsReview
    data_mapper_class = ReviewMapper
    link_field_name = 'dis_instance'
    key_field_name = 'scheme_id'
    create_absent = True

    def create_dis_instance(self, oebs_instance: OebsReview) -> Review:
        return Review()

    def run_rollup(self, object_id=None, dry_run=False):
        super().run_rollup(object_id, dry_run)

        if not dry_run:
            with atomic():
                for reward in Review.objects.filter(oebs_instance__isnull=True):
                    reward.intranet_status = 0
                    reward.save()
