import logging
from datetime import date
from typing import Dict, Optional

from staff.budget_position.models.reward import Reward, RewardCategory
from staff.lib.db import atomic
from staff.lib.sync_tools.rollupper import Rollupper

from staff.oebs.controllers.mappers import DataMapper
from staff.oebs.models import Reward as OebsReward


logger = logging.getLogger(__name__)


class RewardMapper(DataMapper):
    _reward_category_mapping = dict(RewardCategory.choices())

    def __init__(self, object_dict: Dict):
        super().__init__(object_dict)
        self.mapping = (
            ('name', 'name'),
            ('description', 'description'),
            ('start_date', 'start_date'),
            ('end_date', 'end_date'),
            ('category', 'category', self._get_category),
            ('end_date', 'intranet_status', self._get_intranet_status),
            ('scheme_id', 'scheme_id'),
        )

    def _get_category(self, category: Optional[str]) -> Optional[str]:
        return self._reward_category_mapping.get(category)

    @staticmethod
    def _get_intranet_status(end_date: Optional[date]) -> int:
        if end_date is None:
            return 1

        if end_date >= date.today():
            return 1

        return 0


class RewardRollupper(Rollupper):
    model = OebsReward
    data_mapper_class = RewardMapper
    link_field_name = 'dis_instance'
    key_field_name = 'scheme_id'
    create_absent = True

    def create_dis_instance(self, oebs_instance: OebsReward) -> Reward:
        return Reward()

    def run_rollup(self, object_id=None, dry_run=False):
        super().run_rollup(object_id, dry_run)

        if not dry_run:
            with atomic():
                for reward in Reward.objects.filter(oebs_instance__isnull=True):
                    reward.intranet_status = 0
                    reward.save()
