import logging

from django.db.models import F, Q

from staff.budget_position.models import BudgetPosition, ChangeRegistry, Workflow
from staff.budget_position.const import WORKFLOW_STATUS
from staff.departments.controllers.exceptions import ProposalCtlError
from staff.departments.models import HeadcountPosition
from staff.departments.controllers.proposal import ProposalCtl
from staff.lib.db import atomic
from staff.monitorings.models import FailedBpCodesUpdates
from staff.person.models import Staff


logger = logging.getLogger(__name__)


@atomic
def sync_bp_codes_for_persons():
    staff_id_to_code = dict(
        HeadcountPosition.objects
        .filter(current_person__isnull=False, main_assignment=True)
        .exclude(current_person__budget_position__code=F('code'))
        .values_list('current_person_id', 'code')
    )

    logger.info('Start to sync bp codes for persons')

    existing_codes = set(BudgetPosition.objects.values_list('code', flat=True))
    new_codes = set(HeadcountPosition.objects.values_list('code', flat=True)) - existing_codes
    BudgetPosition.objects.bulk_create(BudgetPosition(code=code) for code in new_codes)
    code_to_id = dict(BudgetPosition.objects.values_list('code', 'id'))

    staffs = Staff.objects.filter(id__in=staff_id_to_code.keys())

    changes_filter = Q(id=None)
    for staff in staffs:
        changes_filter |= Q(staff_id=staff.id, budget_position_id=staff.budget_position_id)
    changes_filter &= Q(workflow__status=WORKFLOW_STATUS.PENDING, workflow__proposal_id__isnull=False)

    workflows_to_cancel = []
    staff_to_proposal = {}
    for change in ChangeRegistry.objects.filter(changes_filter).select_related('workflow__proposal', 'staff'):
        workflows_to_cancel.append(change.workflow_id)
        staff_to_proposal[change.person_id] = change.workflow.proposal.proposal_id
        logger.info(
            'Gonna cancel workflow %s for %s in proposal %s',
            str(change.workflow_id), change.staff.login, change.workflow.proposal.proposal_id,
        )

    Workflow.objects.filter(id__in=workflows_to_cancel).update(status=WORKFLOW_STATUS.CANCELLED)

    for staff in staffs:
        bp_id = code_to_id[staff_id_to_code[staff.id]]
        # Делаем update чтобы не триггерить сигнал на создание записи в emission.
        Staff.objects.filter(id=staff.id).update(budget_position_id=bp_id)

        proposal_id = staff_to_proposal.get(staff.id)
        if not proposal_id:
            continue

        try:
            ctl = ProposalCtl(proposal_id=proposal_id)
        except ProposalCtlError:
            logger.exception('Error while trying to find proposal %s', proposal_id)
            FailedBpCodesUpdates.objects.create(
                proposal_id=proposal_id,
                login=staff.login,
                code=staff_id_to_code[staff.id],
            )
            continue

        try:
            with atomic():
                ctl.update_workflow_changes()
        except Exception:
            logger.exception(
                'Error while creating change in proposal %s for %s with new bp code %s',
                proposal_id,
                staff.login,
                staff_id_to_code[staff.id],
            )
            FailedBpCodesUpdates.objects.create(
                proposal_id=proposal_id,
                login=staff.login,
                code=staff_id_to_code[staff.id],
            )

    logger.info('Bp codes for persons synced')
