from django.db import models
from typing import TypeVar, Type

from staff.lib.utils.library import Library
from staff.oebs.controllers.datasources import OEBSDatasourceT

library = Library(
    lambda model: model,
    lambda model: model.__name__,
    lambda model: model.oebs_type,
)


class LinkedManager(models.Manager):
    def __init__(self, linked=True):
        super(LinkedManager, self).__init__()
        self.linked = linked

    def get_queryset(self):
        lookup = {'%s__isnull' % self.model.link_field_name: not self.linked}
        return super(LinkedManager, self).get_queryset().filter(**lookup)


class OEBSModelBase(models.Model):
    created_at = models.DateTimeField(auto_now_add=True, null=True)
    modified_at = models.DateTimeField(auto_now=True, null=True)
    last_sync = models.DateTimeField(
        null=True,
        blank=True,
        verbose_name="Дата выгрузки",
    )
    last_rollup = models.DateTimeField(
        null=True,
        blank=True,
        verbose_name="Дата накатки",
    )
    last_rollup_error = models.TextField(
        blank=True,
        default='',
        verbose_name="Последняя ошибка накатки",
    )
    rollup_warnings = models.TextField(
        blank=True,
        default='',
        verbose_name="Предупреждения накатки",
    )
    st_translation_id = models.IntegerField(
        null=True,
        blank=True,
        verbose_name='id от объекта translation из startrek'
    )

    link_field_name = ''

    objects = models.Manager()
    linked = LinkedManager()
    unlinked = LinkedManager(linked=False)
    oebs_type = None
    method = None
    datasource_class = None  # type: Type[OEBSDatasourceT]

    @property
    def dis_object(self):
        return getattr(self, self.link_field_name)

    class Meta:
        abstract = True


OEBSModelBaseT = TypeVar('OEBSModelBaseT', bound=OEBSModelBase)
