from typing import Type

from staff.celery_app import app
from staff.lib.tasks import LockedTask
from staff.oebs.models import (
    Bonus,
    BusinessCenter,
    Currency,
    Employee,
    Geography,
    HRProduct,
    HeadcountPosition,
    Job,
    LeaveBalance,
    OEBSModelBaseT,
    Occupation,
    Office,
    Organization,
    PaySys,
    PersonOccupation,
    Review,
    Reward,
)
from staff.oebs.models.oebs_headcount_position import OebsHeadcountPosition
from staff.oebs.tasks.link_tasks import LinkTask
from staff.oebs.tasks.sync_tasks import SyncTask
from staff.oebs.tasks.rollup_tasks import RollupTask


class PullModel(LockedTask):
    model: Type[OEBSModelBaseT] = None
    sync: bool = False
    link: bool = False
    rollup: bool = False

    def locked_run(self, *args, **kwargs):
        model = self.model.__name__
        if self.sync:
            SyncTask(model=model)
        if self.link:
            LinkTask(model=model)
        if self.rollup:
            RollupTask(model=model)

    def get_lock_name(self, *args, **kwargs):
        name = self.__class__.__name__
        model = self.model.__name__ if self.model else ''
        return '{}_{}'.format(name, model)


@app.task(ignore_result=True)
class PullEmployee(PullModel):
    model = Employee
    sync = True
    link = True
    rollup = True


@app.task(ignore_result=True)
class PullLeaveBalance(PullModel):
    model = LeaveBalance
    sync = True
    link = True
    rollup = True


@app.task(ignore_result=True)
class PullOffice(PullModel):
    model = Office
    sync = True
    link = False
    rollup = True


@app.task(ignore_result=True)
class PullOrganization(PullModel):
    model = Organization
    sync = True
    link = False
    rollup = True


@app.task(ignore_result=True)
class PullHeadcounts(PullModel):
    model = HeadcountPosition
    sync = True
    rollup = True


@app.task(ignore_result=True)
class PullBudgetPositionAssignment(PullModel):
    model = OebsHeadcountPosition
    sync = True
    link = False
    rollup = True


@app.task(ignore_result=True)
class PullOccupation(PullModel):
    model = Occupation
    sync = True
    link = False
    rollup = True


@app.task(ignore_result=True)
class PullPersonOccupation(PullModel):
    model = PersonOccupation
    sync = True
    link = True
    rollup = True


@app.task(ignore_result=True)
class PullJobs(PullModel):
    model = Job
    sync = True
    link = False
    rollup = False


@app.task(ignore_result=True)
class PullGeography(PullModel):
    model = Geography
    sync = True
    link = True
    rollup = True


@app.task(ignore_result=True)
class PullBonus(PullModel):
    model = Bonus
    sync = True
    link = True
    rollup = True


@app.task(ignore_result=True)
class PullReward(PullModel):
    model = Reward
    sync = True
    link = True
    rollup = True


@app.task(ignore_result=True)
class PullPaySys(PullModel):
    model = PaySys
    sync = True
    link = False
    rollup = False


@app.task(ignore_result=True)
class PullReview(PullModel):
    model = Review
    sync = True
    link = True
    rollup = True


@app.task(ignore_result=True)
class PullHRProduct(PullModel):
    model = HRProduct
    sync = True
    link = True
    rollup = True


@app.task(ignore_result=True)
class PullBusinessCenter(PullModel):
    model = BusinessCenter
    sync = True
    link = False
    rollup = True


@app.task(ignore_result=True)
class PullCurrency(PullModel):
    model = Currency
    sync = True
    link = False
    rollup = True
