import logging

from typing import Type

from staff.celery_app import app

from staff.lib import waffle
from staff.lib.sync_tools.rollupper import RollupperT
from staff.lib.tasks import LockedTask
from staff.oebs.controllers.rolluppers import rolluppers_library
from staff.oebs.models import OEBSModelBaseT, library


logger = logging.getLogger(__name__)


@app.task(ignore_result=True)
class RollupTask(LockedTask):
    def get_rollupper(self, model: Type[OEBSModelBaseT]) -> Type[RollupperT]:
        return rolluppers_library[model]

    def locked_run(self, *args, **kwargs):
        model = kwargs.get('model')
        if not model:
            logger.error('Model argument is required')
            return

        logger.info('Rolling up model {}'.format(model))
        object_id = kwargs.get('object_id')
        dry = kwargs.get('dry', False)
        create_absent = (
            kwargs.get('create_absent', False) or
            waffle.switch_is_active('rollup_oebs_pull_create_%s' % model.lower())
        )
        force = kwargs.get('force', False)

        rollupper = self.get_rollupper(library[model])

        switch = 'rollup_oebs_%s' % model.lower()
        if waffle.switch_is_active(switch) or force:
            rollupper.rollup(object_id=object_id, dry_run=dry, create_absent=create_absent)
        else:
            logger.info('%s is disabled', switch)

    def get_lock_name(self, *args, **kwargs):
        name = self.__class__.__name__
        model = kwargs.get('model')
        return '{}_{}'.format(name, model)
