import logging

from typing import Type, AnyStr, Union

from staff.celery_app import app
from staff.lib.tasks import LockedTask
from staff.lib.sync_tools import Updater

from staff.oebs.controllers import updaters, datagenerators
from staff.oebs.models import library, OEBSModelBaseT, Office


logger = logging.getLogger(__name__)


@app.task(ignore_result=True)
class SyncTask(LockedTask):
    @staticmethod
    def _create_data_diff_merger(model: Type[OEBSModelBaseT], data_generator):
        if model == Office:
            return updaters.OEBSPlacementDataDiffMerger(data_generator, logger)

        return updaters.OEBSDataDiffMerger(data_generator, logger)

    def _create_updater(self, model: Type[OEBSModelBaseT], object_id) -> Updater:
        datasource = model.datasource_class(object_type=model.oebs_type, method=model.method)
        data_generator_class = datagenerators.library[model]
        data_generator = data_generator_class(datasource)
        data_diff_merger = self._create_data_diff_merger(model, data_generator)

        result = updaters.OEBSUpdater(data_diff_merger, logger)
        result.source_type = 'OEBS ' + model.__name__
        if object_id:
            result.do_delete = False

        return result

    def locked_run(self, model: Union[AnyStr, Type[OEBSModelBaseT]] = None, delete: bool = False, **kwargs):
        logger.debug('SyncTask {0}'.format(locals()))
        model: Type[OEBSModelBaseT] = library[model]
        logging.info('Syncing model {}'.format(model))

        if delete:
            logger.info('Deleting all of OEBS{name}'.format(name=model.__name__))
            model.objects.all().delete()

        self._create_updater(model, kwargs.get('object_id')).run_sync()

    def get_lock_name(self, *args, **kwargs):
        name = self.__class__.__name__
        model = kwargs.get('model')
        return '{}_{}'.format(name, model)
