import datetime
import json
import logging

from django.conf import settings

from staff.lib.tvm2 import TVM_SERVICE_TICKET_HEADER, get_tvm_ticket_by_deploy
from staff.lib import requests


logger = logging.getLogger(__name__)

GET_ASSIGNMENT_TIMEOUTS = (1, 2, 4)  # sec


def _sort_person_assignments(assignments_list):
    def primary_flag_first(item):
        return {'Y': 0}.get(item.get('primaryFlag'), 1)

    def legislation_code_first(item):
        return {'RU': 0}.get(item.get('legislationCode'), 1)

    assignments_list.sort(
        key=lambda item: (primary_flag_first(item), legislation_code_first(item))
    )


def get_oebs_assignments_by_person_logins(person_logins, data_date=None):
    """{login: [
        {
            'assignmentID': 73220,
            'legislationCode': 'RU',
            'primaryFlag': 'Y',
            'taxUnitID': 121,
            'taxUnitName': 'ООО Яндекс',
            'orgID': 77652,
            'orgName': 'Группа разработки Стаффа и Фемиды',
            'contractType': 'Основное место работы',
        },
        ...
    ]}"""
    if data_date is None:
        data_date = datetime.datetime.utcnow()

    date_str = data_date.strftime('%Y-%m-%d')
    request_data = json.dumps({'login': person_logins, 'date': date_str})

    headers = {
        'Content-type': 'application/json',
        TVM_SERVICE_TICKET_HEADER: get_tvm_ticket_by_deploy('oebs-api'),
    }

    try:
        response = requests.post(
            url=settings.OEBS_ASSIGNMENTS_URL,
            data=request_data,
            headers=headers,
            timeout=GET_ASSIGNMENT_TIMEOUTS,
            log_message=request_data,
        )
    except requests.Timeout:
        logger.info("OEBS is not responding for logins %s", person_logins, exc_info=True)
        return {}
    except Exception:
        logger.exception('Request for logins %s failed', person_logins)
        return {}

    if response.status_code != 200:
        logger.info('Request for logins %s failed with oebs status code: %s', person_logins, response.status_code)
        return {}

    if not response.content:
        logger.info('Empty response for logins %s', person_logins)
        return {}

    assignments_data = []
    try:
        assignments_data = response.json()['logins']
    except (ValueError, KeyError):
        logger.exception('Wrong OEBS response, logins: %s, response: %s', person_logins, response.text)

    assignments = dict.fromkeys(login.lower() for login in person_logins)

    for assignment_data in assignments_data:
        login = assignment_data['login'].lower()
        assignments[login] = assignment_data['assignments']
        _sort_person_assignments(assignments[login])

    return assignments
