import logging

from django.conf import settings
from django.http import HttpResponseForbidden

from staff.lib import requests, tvm2
from staff.lib.auth import auth_mechanisms as auth
from staff.lib.decorators import available_by_tvm, responding_json


logger = logging.getLogger(__name__)

OEBS_TVM_NAME = 'oebs-hrproc'


@responding_json
def get_loan_data(request):
    try:
        response = requests.get(
            url=f'{settings.REVIEW_URL}/v1/loan/info/',
            headers=tvm2.get_tvm_headers_for_requests(request, 'review'),
            timeout=(2, 5, 10),
        )
        response.raise_for_status()
        return response.json()
    except Exception:
        logger.warning('Failed to get loan/info from review', exc_info=True)
        return {'error': 'Review not available'}


@available_by_tvm([OEBS_TVM_NAME])
@responding_json
def get_loan_available(request, login: str):
    if request.auth_mechanism != auth.TVM and request.user.username != login:
        return HttpResponseForbidden()

    headers = tvm2.get_tvm_headers_for_requests(request, 'review')
    is_from_oebs = (
        OEBS_TVM_NAME == tvm2.get_calling_service_name(request)
        or request.user.username in settings.LOAN_OEBS_ROBOTS
    )
    if is_from_oebs:
        # HACK
        # Review-service expects user to receive data
        # Need to be changed after https://st.yandex-team.ru/CIA-2808
        headers[tvm2.TVM_USER_TICKET_HEADER] = tvm2.get_user_ticket_for_robot_staff()

    params = dict(request.GET or request.POST or {})
    params['person'] = login
    try:
        response = requests.get(
            url=f'{settings.REVIEW_URL}/v1/loan/requirements/',
            headers=headers,
            timeout=(2, 5, 10),
            params=params,
        )
        response.raise_for_status()
        return response.json()
    except Exception:
        logger.warning('Failed to get loan/requirements from review', exc_info=True)
        return {'error': 'Review not available'}
