import logging
import re
import unicodedata
from datetime import timedelta
from typing import Any, Dict, List, Tuple, Optional

import ldap
import requests
from django.conf import settings

from staff.lib.db import atomic
from staff.lib.ldap_helpers import encode_str, from_ad_time, ldap_search, LDAPContext, LDAPException
from staff.lib.models.roles_chain import chiefs_chain_for_person

from staff.departments.models.department import DepartmentRoles

from staff.person.models import Staff, PersonADInformation


logger = logging.getLogger(__name__)

PERSON_ACTIVEDIRECTORY_FIELDS = ('has_exchange', 'password_expires_at', 'passwd_set_at')


def _convert_phone(number: Optional[str]):
    if not number or len(number) > settings.MAX_WORK_PHONE_LEN or not number.isnumeric():
        return None

    return int(number)


def _has_exchange_mail(dn_search: Tuple[str, ...], login: str) -> bool:
    fields_map_check_ou = {
        'msExchMailboxGuid': lambda v: ('ms_exch_mailbox_guid_is_empty', v is None),
        'homeMDB': lambda v: ('home_mdb_is_empty', v is None),
    }
    data: Dict[str, Any] = ldap_search(dn_search, '(sAMAccountName=%s)' % login, fields_map_check_ou)

    if data.get('ms_exch_mailbox_guid_is_empty'):
        return False

    if data.get('home_mdb_is_empty'):
        return False

    return True


def get_ad_person_data(login):
    search_dn = (
        'CN=Users,DC=ld,DC=yandex,DC=ru',
        'OU=ForeignUsers,DC=ld,DC=yandex,DC=ru',
        'OU=TechUsers,DC=ld,DC=yandex,DC=ru',
        'OU=OLD Users,DC=ld,DC=yandex,DC=ru',
    )
    query = '(sAMAccountName=%s)' % login
    fields_map = {
        'msDS-UserPasswordExpiryTimeComputed': lambda v: ('password_expires_at', v and from_ad_time(v[0])),
        'pwdLastSet': lambda v: ('passwd_set_at', v and from_ad_time(v[0])),
        'telephoneNumber': lambda v: ('work_phone', v and _convert_phone(v[0])),
        'mail': lambda v: ('has_exchange', _has_exchange_mail(search_dn, login))
    }
    return ldap_search(search_dn, query, fields_map)


def _get_person_attributes_diff(new_values: Dict[str, Any], person_ctl) -> Dict[str, Any]:
    diff = {
        field: value
        for field, value in new_values.items()
        if hasattr(person_ctl, field) and getattr(person_ctl, field) != value
    }
    return diff


def _update_person_active_directory_data(
    new_values: Dict[str, Any],
    person_ad_info_model: PersonADInformation,
) -> None:
    update_args = {}
    for field in PERSON_ACTIVEDIRECTORY_FIELDS:
        if getattr(person_ad_info_model, field) != new_values[field]:
            update_args[field] = new_values[field]

    if update_args:
        PersonADInformation.objects.filter(id=person_ad_info_model.id).update(**update_args)


def update_person_from_ad(person: Staff) -> None:
    from staff.person.controllers import PersonCtl

    try:
        ad_data = get_ad_person_data(person.login)
        person_ad_info_model, _ = PersonADInformation.objects.get_or_create(person_id=person.id)
        _update_person_active_directory_data(ad_data, person_ad_info_model)

        if not _get_person_attributes_diff(ad_data, PersonCtl(person)):
            return

        logger.info('Found diff')
        with atomic():
            person_ctl = PersonCtl(Staff.objects.select_for_update().get(login=person.login))
            diff = _get_person_attributes_diff(ad_data, person_ctl)
            if not diff:
                return

            logger.info('Person %s updating %s...', person.login, list(diff.keys()))
            person_ctl.update(diff).save()

    except (LDAPException, TypeError):
        logger.exception('Error trying to get work_phone from AD for user %s', person.login)


POSTAL_CODE_RGX = re.compile(r'[0-9]{5,7}')

SEARCH_DN = (
    'CN=Users,DC=ld,DC=yandex,DC=ru',
    'OU=ForeignUsers,DC=ld,DC=yandex,DC=ru',
    'OU=TechUsers,DC=ld,DC=yandex,DC=ru',
)


def _get_employee_type(person: Staff) -> str:
    affiliation_to_ad_type = {
        'yandex': 'yandex',
        'external': 'outstaff',
    }

    if person.is_robot:
        return 'robot'

    result = affiliation_to_ad_type.get(person.affiliation)
    if not result:
        logger.error('AD affiliation type not found for user %s', person.login)
    return result


def _get_preferred_language(person: Staff) -> Optional[str]:
    preferred_language_map = {
        'ru': 'ru-RU',
        'en': 'en-US',
    }

    result = preferred_language_map.get(person.lang_ui)

    if not result:
        logger.info('AD preferred language %s not detected for %s', person.lang_ui, person.login)

    return result


def _remove_accents(value: Optional[str]) -> Optional[str]:
    if not value:
        return value

    normalized_form = ''
    for ch in value:
        if ch in ('Ё', 'ё', 'Й', 'й'):
            normalized_form += ch
        else:
            normalized_form += unicodedata.normalize('NFKD', ch)

    return u''.join([ch for ch in normalized_form if not unicodedata.combining(ch)])


def _get_person_data(person: Staff) -> Dict[str, str]:
    try:
        postal_code = POSTAL_CODE_RGX.findall(f'{person.office.address1_en} {person.office.address2_en}')[0]
    except IndexError:
        postal_code = ''

    return {
        'department': person.department.name_en,
        'title': person.position_en,
        'streetAddress': person.office.address1_en,
        'co': person.office.city.country.name_en,
        'l': person.office.city.name_en,
        'postalCode': postal_code,
        'physicalDeliveryOfficeName': person.office.name_en,
        'employeeType': _get_employee_type(person),
        'extensionAttribute3': person.work_email,
        'yaFirstNameEn': _remove_accents(person.first_name_en),
        'yaLastNameEn': _remove_accents(person.last_name_en),
        'yaFirstNameRu': _remove_accents(person.first_name),
        'yaLastNameRu': _remove_accents(person.last_name),
        'preferredLanguage': _get_preferred_language(person),
    }


def _get_modlist(person: Staff, fields_map: Dict[str, callable]) -> List[Tuple[int, str, bytes]]:
    modlist = [
        (ldap.MOD_REPLACE, field, encode_str(value)[:64] or None)
        for field, value in _get_person_data(person).items()
    ]

    is_general_director = any(
        role.id == DepartmentRoles.GENERAL_DIRECTOR.value
        for role in person.department_roles.all()
    )
    if is_general_director:
        modlist.append((ldap.MOD_REPLACE, 'manager', None))
    else:
        chiefs_chain = chiefs_chain_for_person(
            person=person,
            fields=['login'],
            roles=[DepartmentRoles.CHIEF.value, DepartmentRoles.GENERAL_DIRECTOR.value]
        )
        if chiefs_chain:
            chief_login = chiefs_chain[0]['login']
            chief_query = '(sAMAccountName=%s)' % chief_login
            chief_dn = ldap_search(SEARCH_DN, chief_query, fields_map, ldap.SCOPE_SUBTREE)['distinguishedName']
            modlist.append((ldap.MOD_REPLACE, 'manager', encode_str(chief_dn)))
    return modlist


def update_ad_user_data(person: Staff) -> None:
    fields_map = {'distinguishedName': lambda v: ('distinguishedName', v[0])}
    person_query = '(sAMAccountName=%s)' % person.login
    person_dn = ldap_search(SEARCH_DN, person_query, fields_map, ldap.SCOPE_SUBTREE)['distinguishedName']

    if not person_dn:
        logger.error('User %s not found in AD', person.login)
        return

    with LDAPContext(settings.LDAP_HOST, user=settings.LDAP_STAFF_USER, password=settings.ROBOT_STAFF_PASSWORD) as ad:
        ad.modify_s(
            dn=person_dn,
            modlist=_get_modlist(person, fields_map),
        )


def update_ad_user_avatar(person: Staff) -> None:
    avatar_size = 100

    response = requests.get(settings.CENTER_PERSON_AVATAR_URL % (person.login, avatar_size))
    if response.status_code == 200:
        fields_map = {'distinguishedName': lambda v: ('distinguishedName', v[0])}
        person_query = '(sAMAccountName=%s)' % person.login
        person_dn = ldap_search(SEARCH_DN, person_query, fields_map, ldap.SCOPE_SUBTREE)['distinguishedName']

        modlist = [(ldap.MOD_REPLACE, 'thumbnailPhoto', encode_str(response.content))]
        with LDAPContext(
                dc=settings.LDAP_HOST,
                user=settings.LDAP_STAFF_USER,
                password=settings.ROBOT_STAFF_PASSWORD,
        ) as ad:
            ad.modify_s(
                dn=person_dn,
                modlist=modlist,
            )


def _convert_max_password_age(age):
    return timedelta(microseconds=-int(age) // 10)


def get_ad_domain_data():
    fields_map = {
        'maxPwdAge': lambda v: (
            'max_password_age', _convert_max_password_age(v[0])
        ),
    }
    search_dn = ('DC=ld,DC=yandex,DC=ru',)
    query = '(objectcategory=domainDNS)'
    return ldap_search(search_dn, query, fields_map, ldap.SCOPE_BASE)


def get_max_password_age():
    max_password_age = getattr(get_max_password_age, '_cached', None)
    if max_password_age is None:
        try:
            domain_data = get_ad_domain_data()
            max_password_age = domain_data['max_password_age']
        except (LDAPException, TypeError):
            max_password_age = timedelta(93)
        setattr(get_max_password_age, '_cached', max_password_age)

    return max_password_age
