from datetime import datetime

import yenv
from django.contrib import admin
from django import forms

from staff.lib.formfields import StaffAutocompleteField
from staff.lib.utils.admin import StaffModelAdmin, BaseIntranetAdmin
from staff.keys.models import SSHKey

from staff.person.forms import StaffExtraFieldsAdminForm
from staff.person.models import (
    Contact,
    ContactType,
    MemorialProfile,
    Occupation,
    PaymentSystem,
    Staff,
    StaffCar,
    StaffExtraFields,
    StaffPhone,
    WelcomeEmailTemplate,
)
from staff.person.passport.internal import IntPassport


class StaffExtraFieldsAdmin(StaffModelAdmin):
    list_display = (
        'staff',
        'nda_end_date',
        'contract_end_date',
        'byod_access',
    )
    readonly_fields = (
        'main_img_for_upload',
        'avatar_img_for_upload',
        'gravatar_img_for_upload',
    )
    form = StaffExtraFieldsAdminForm


class ContactAdmin(StaffModelAdmin):
    staff_field = 'person'

    list_display = (
        'person',
        'contact_type',
        'private',
        'account_id',
        'position',
    )
    ordering = ('person', 'position')


class ContactTypeAdmin(admin.ModelAdmin):
    list_display = (
        'name',
        'url_pattern',
        'validation_type',
        'show_in_vcard',
    )
    ordering = ('position', )


class PaymentSystemAdmin(admin.ModelAdmin):
    list_display = (
        'name',
        'description',
    )


class OccupationAdmin(BaseIntranetAdmin):
    list_display = (
        'name',
        'description',
        'description_en',
        'code',
        'group_femida',
        'group_review',
        'group_bonus',
        'group_reward',
    )
    list_editable = (
        'description',
        'description_en',
        'code',
        'group_femida',
        'group_review',
        'group_bonus',
        'group_reward',
    )


# Inlines for staff-satellite models
class SSHKeyInline(admin.TabularInline):
    model = SSHKey
    extra = 1


class StaffCarInline(admin.TabularInline):
    model = StaffCar
    extra = 1
    exclude = ('native_lang', 'intranet_status')


class StaffPhoneModelForm(forms.ModelForm):
    def __init__(self, *args, **kwargs):
        super(StaffPhoneModelForm, self).__init__(*args, **kwargs)
        for field in self.fields.values():
            field.required = False

    class Meta:
        fields = '__all__'
        models = StaffPhone


class StaffPhoneInline(admin.TabularInline):
    model = StaffPhone
    form = StaffPhoneModelForm
    extra = 1
    exclude = ('native_lang', 'intranet_status')


class StaffModelForm(forms.ModelForm):
    def __init__(self, *args, **kwargs):
        super(StaffModelForm, self).__init__(*args, **kwargs)
        for field in self.fields.values():
            field.required = False

    def clean_login_passport(self):
        # Because login_passport = CharField(null=True, unique=True)
        return self.cleaned_data['login_passport'] or None

    class Meta:
        fields = '__all__'
        models = Staff


def revert_firing(modeladmin, request, queryset):
    for person in queryset:
        person.is_dismissed = False
        person.quit_at = None
        person.save()
        if yenv.type == 'production':
            passport = IntPassport(uid=person.uid)
            passport.unblock()


revert_firing.short_description = "Revert firing"


# External models admins
class StaffAdmin(StaffModelAdmin):
    form = StaffModelForm
    list_display = (
        '__str__',
        'login',
        'email',
        'department',
        'uid',
        'guid',
        'login_passport',
        'budget_position',
    )
    readonly_fields = (
        'budget_position',
    )
    search_fields = StaffModelAdmin.STAFF_SEARCH_FIELDS
    inlines = (StaffCarInline, StaffPhoneInline, SSHKeyInline)

    fieldsets = (
        ('ID', {'fields': (
            ('login', 'normal_login'),
            ('uid', 'guid'),
        )}),
        ('Name', {'fields': (
            ('last_name', 'last_name_en'),
            ('first_name', 'first_name_en'),
            'middle_name',
            'has_namesake',
        )}),
        ('Settings', {'classes': ('collapse', ), 'fields': (
            'tz',
            'lang_ui',
            'hide_middle_name',
            'show_all_middle_name',
            'shell',
        )}),
        ('Flags', {'classes': ('collapse', ), 'fields': (
            'affiliation',
            'is_homeworker',
            'is_robot',
            'is_big_boss',
            'is_dismissed',
        )}),
        ('Official', {'classes': ('collapse', ), 'fields': (
            'department',
            'organization',
            ('position', 'position_en'),
            ('duties', 'duties_en'),
            'employment',
            ('join_at', 'quit_at'),
            'vacation',
            'extra_vacation',
            'date_completion_internship',
        )}),
        ('Official Contacts', {'classes': ('collapse', ), 'fields': (
            ('login_mail', 'domain'),
            'work_email',
            'work_phone',
            'login_crm',
            'has_exchange',
        )}),
        ('Location', {'classes': ('collapse', ), 'fields': (
            'office',
            'table',
            ('location_descr', 'location_descr_en'),
        )}),
        ('Personal', {'classes': ('collapse', ), 'fields': (
            'birthday',
            'gender',
            'family_status',
            'children',
            'tshirt_size',
            'mobile_phone_model',
            ('car', 'car_num'),
            ('address', 'address_en'),
            'about',
        )}),
        ('Education', {'classes': ('collapse', ), 'fields': (
            'edu_status',
            'edu_direction',
            'edu_place',
            'edu_place_en',
            'edu_date',
        )}),
        ('Contacts', {'classes': ('collapse', ), 'fields': (
            'home_email',
            'home_phone',
            'home_page',
            'icq',
            'jabber',
            'login_skype',
            'login_mk',
            'login_twitter',
            'login_lj',
            'mobile_phone',
            ('login_passport', 'is_login_passport_confirmed'),
        )}),
        ('Old', {'classes': ('collapse', ), 'fields': (
            'from_staff_id',
            'login_ld',
            'en_name',
            'computer',
            'lang_content',
        )}),
    )

    actions = [revert_firing]


class SSHKeyAdmin(StaffModelAdmin):
    list_display = ('staff', 'description', 'key')
    ordering = ('staff__id', 'description')
    # list_editable = ('description', )


class StaffCarAdmin(StaffModelAdmin):
    list_display = ('staff', 'model', 'plate')


class StaffPhoneAdmin(StaffModelAdmin):
    search_fields = ('number', )
    list_select_related = True
    ordering = 'staff',

    def login(self, obj):
        return obj.staff.login

    list_display = (
        'login',
        'staff',
        'number',
        'description',
        'type',
    )

    def save_model(self, request, obj, form, change):
        now = datetime.now()
        if not change:
            obj.created_at = now
        obj.modified_at = now
        super(StaffPhoneAdmin, self).save_model(request, obj, form, change)


class MemorialAdmin(StaffModelAdmin):
    staff_field = 'person'

    def dates(self, obj):
        if not obj.person.birthday:
            return f'???? — {obj.death_date.year}'

        return f'{obj.person.birthday.year} — {obj.death_date.year}'

    def login(self, obj):
        return obj.person.login

    def name(self, obj):
        return str(obj.person)

    def is_enabled(self, obj):
        return obj.intranet_status > 0

    list_display = (
        'login',
        'person',
        'dates',
        'is_enabled',
    )

    def formfield_for_foreignkey(self, db_field, request, **kwargs):
        if db_field.name == 'person':
            return StaffAutocompleteField(include_dismissed=True)

        return super(MemorialAdmin, self).formfield_for_foreignkey(
            db_field, request, **kwargs)


class WelcomeEmailTemplateAdmin(admin.ModelAdmin):
    list_display = ('id', 'text', 'text_en')
    list_display_links = list_display


admin.site.register(Contact, ContactAdmin)
admin.site.register(ContactType, ContactTypeAdmin)
admin.site.register(MemorialProfile, MemorialAdmin)
admin.site.register(Occupation, OccupationAdmin)
admin.site.register(PaymentSystem, PaymentSystemAdmin)
admin.site.register(SSHKey, SSHKeyAdmin)
admin.site.register(Staff, StaffAdmin)
admin.site.register(StaffCar, StaffCarAdmin)
admin.site.register(StaffExtraFields, StaffExtraFieldsAdmin)
admin.site.register(StaffPhone, StaffPhoneAdmin)
admin.site.register(WelcomeEmailTemplate, WelcomeEmailTemplateAdmin)
