import re

from itertools import groupby

from staff.person.models import Staff

from staff.lib.utils.dict import invert_dict
from staff.person_profile.controllers.contacts import SYNC_IDS
from staff.person_profile.forms.contacts import ContactForm

from staff.person.models import Contact

OLD_2_NEW = invert_dict(SYNC_IDS)
OLD_2_NEW['login_lj'] = [8]

TYPES_ORDER = [
    9,  # Ещё рабочая почта
    4,  # Skype
    2,  # Jabber
    10,  # GitHub
    1,  # Личная почта
    7,  # Личный сайт
    6,  # Мой Круг
    11,  # Facebook
    12,  # Вконтакте
    5,  # Twitter
    13,  # Хабрахабр
    14,  # Instagram
    15,  # Flickr
    8,  # Живой журнал
    16,  # Tumblr
    17,  # Blogspot
    3,  # ICQ
]

RE_SCHEME = re.compile(r'^https?://')


def contacts_to_type_dict(contacts):
    if not contacts:
        return {}
    return {
        type_id: list(contacts)
        for type_id, contacts
        in groupby(contacts, lambda contact: contact.contact_type_id)
    }


def has_value_in_contacts(value, contacts):
    if contacts:
        return any(contact.account_id == value for contact in contacts)
    return False


def get_all_contacts_dict():
    return {
        person_id: list(contacts)
        for person_id, contacts
        in groupby(
            Contact.objects.order_by('person__id', 'contact_type__id', 'id'),
            lambda contact: contact.person_id
        )
    }


class ContactTypeMatcher(object):

    def guess_contact(self, url):
        for key in dir(self):
            if key.startswith('_match_'):
                result = getattr(self, key)(url)
                if result:
                    return result
        return self._default_guess(url)

    def _new_contact(self, contact_type_id, account_id):
        return {
            'contact_type_id': contact_type_id,
            'account_id': account_id,
        }

    def _default_guess(self, url):
        if not RE_SCHEME.match(url):
            url = 'http://' + url
        return self._new_contact(7, url)

    def _match_personal_email(self, url):
        _re = r'(.+@.+)'
        match = re.match(_re, url)
        if match:
            return self._new_contact(1, match.groups()[0])

    def _match_lj_0(self, url):
        _re = r'^(?:http[s]?://)?(?:www\.)?(.*)\.livejournal\.com/?$'
        match = re.match(_re, url)
        if match:
            return self._new_contact(8, match.groups()[0])

    def _match_lj_1(self, url):
        _re = r'^(?:http[s]?://)?(?:www\.)?livejournal\.com/users/([^/]*)/?$'
        match = re.match(_re, url)
        if match:
            return self._new_contact(8, match.groups()[0])

    def _match_lj_2(self, url):
        _re = r'^([a-zA-Z\d]+\w*[a-zA-Z\d]+)$'
        match = re.match(_re, url)
        if match:
            return self._new_contact(8, match.groups()[0])

    def _match_vk_0(self, url):
        _re = r'^(?:http[s]?://)?(?:www\.)?vk\.com/([^/]*)/?$'
        match = re.match(_re, url)
        if match:
            return self._new_contact(12, match.groups()[0])

    def _match_vk_1(self, url):
        _re = r'^(?:http[s]?://)?(?:www\.)?vkontakte\.ru/([^/]*)/?$'
        match = re.match(_re, url)
        if match:
            return self._new_contact(12, match.groups()[0])

    def _match_facebook_0(self, url):
        _re = r'^(?:http[s]?://)?(?:www\.)?(?:fb|facebook)\.com/([^/]*)/?$'
        match = re.match(_re, url)
        if match:
            return self._new_contact(11, match.groups()[0])

    def _match_blogspot_0(self, url):
        _re = r'^(?:http[s]?://)?(.*)\.blogspot\.com/?$'
        match = re.match(_re, url)
        if match:
            return self._new_contact(17, match.groups()[0])

    def _match_instagram_0(self, url):
        _re = r'^(?:http[s]?://)?(?:www\.)?instagram\.com/([^/]*)/?$'
        match = re.match(_re, url)
        if match:
            return self._new_contact(14, match.groups()[0])

    def _match_habrahabr_0(self, url):
        _re = r'^(?:http[s]?://)?(?:www\.)?habrahabr\.ru/users/([^/]*)/?'
        match = re.match(_re, url)
        if match:
            return self._new_contact(13, match.groups()[0])

    def _match_habrahabr_1(self, url):
        _re = r'^(?:http[s]?://)?(.*)\.habrahabr\.ru/?$'
        match = re.match(_re, url)
        if match:
            return self._new_contact(13, match.groups()[0])

    def _match_flickr_0(self, url):
        _re = r'^(?:http[s]?://)?(?:www\.)?flickr\.com/photos/([^/]*)/?$'
        match = re.match(_re, url)
        if match:
            return self._new_contact(15, match.groups()[0])

    def _match_tumblr_0(self, url):
        _re = r'^(?:http[s]?://)?(.*)\.tumblr\.com/?$'
        match = re.match(_re, url)
        if match:
            return self._new_contact(16, match.groups()[0])

    def _match_github_0(self, url):
        _re = r'^(?:http[s]?://)?(.*)\.github\.com/?$'
        match = re.match(_re, url)
        if match:
            return self._new_contact(10, match.groups()[0])

    def _match_github_1(self, url):
        _re = r'^(?:http[s]?://)?(?:www\.)?github\.com/([^/]*)/?$'
        match = re.match(_re, url)
        if match:
            return self._new_contact(10, match.groups()[0])

    def _match_moikrug_0(self, url):
        _re = r'^(?:http[s]?://)?(.*)\.moikrug\.ru/?$'
        match = re.match(_re, url)
        if match:
            return self._new_contact(6, match.groups()[0])

    def _match_twitter_0(self, url):
        _re = r'^(?:http[s]?://)?(?:www\.)?twitter\.com/([^/]*)/?$'
        match = re.match(_re, url)
        if match:
            return self._new_contact(5, match.groups()[0])


CTM = ContactTypeMatcher()


class Migrator(object):

    def __init__(self):
        self._contacts = {}

    def migrate_all(self):
        all_persons = Staff.objects.values(
            'id',
            'is_dismissed',
            'home_email',
            'home_page',
            'icq',
            'jabber',
            'login_skype',
            'login_mk',
            'login_twitter',
            'login_lj',
        )
        contacts_dict = get_all_contacts_dict()
        for data in all_persons:
            self.migrate_one(data, contacts_dict.get(data['id'], []))

    def migrate_one(self, data, contacts):
        self._contacts = {}
        meta = {
            'person_id': data.pop('id'),
            'is_dismissed': data.pop('is_dismissed'),
        }
        contacts_dict = contacts_to_type_dict(contacts)

        for field, value in data.items():
            if not hasattr(self, field):
                self._common(field, value, contacts_dict, meta)
            else:
                getattr(self, field)(field, value, contacts_dict, meta)

        position = 0
        for type_id in TYPES_ORDER:
            contacts = self._contacts.get(type_id)
            if contacts:
                for contact in contacts:
                    position += 1
                    contact.position = position
                    contact.save()

    def _common(self, field, value, contacts_dict, meta, data=None):
        if not value:
            return
        person_id = meta['person_id']
        contact_type_id = data['contact_type_id'] if data else OLD_2_NEW.get(field)[0]
        contacts = contacts_dict.get(contact_type_id)
        value = data['account_id'] if data else value

        if not has_value_in_contacts(value, contacts):
            form = ContactForm(data={
                'contact_type': contact_type_id,
                'account_id': value,
            })
            if form.is_valid():
                _data = form.cleaned_data
                self._contacts.setdefault(contact_type_id, []).append(
                    Contact(
                        person_id=person_id,
                        contact_type_id=_data['contact_type'].id,
                        account_id=_data['account_id'],
                    )
                )
            else:
                print('ID[{0}, is_dismissed = {1}]: {2} = {3}'.format(
                    person_id,
                    meta['is_dismissed'],
                    contact_type_id,
                    value
                ))

    def login_lj(self, field, value, contacts_dict, meta):
        contact = CTM.guess_contact(value)
        if contact:
            self._common('', value, contacts_dict, meta, data=contact)

    def icq(self, field, value, contacts_dict, meta):
        value = value.replace(' ', '')
        if value:
            self._common('icq', value, contacts_dict, meta)

    def home_email(self, field, value, contacts_dict, meta):
        emails = re.split(r'[,; ]+', value)
        if emails:
            for email in emails:
                self._common('home_email', email, contacts_dict, meta)

    def home_page(self, field, value, contacts_dict, meta):
        if value:
            if not RE_SCHEME.match(value):
                value = 'http://' + value
            self._common('home_page', value, contacts_dict, meta)

    def login_mk(self, field, value, contacts_dict, meta):
        contact = CTM._match_moikrug_0(value)
        if contact:
            self._common('', value, contacts_dict, meta, data=contact)
        else:
            self._common('login_mk', value, contacts_dict, meta)

    def login_twitter(self, field, value, contacts_dict, meta):
        contact = CTM._match_twitter_0(value)
        if contact:
            self._common('', value, contacts_dict, meta, data=contact)
        else:
            self._common('login_twitter', value, contacts_dict, meta)
