from itertools import groupby

from django.db.models import F

from staff.person.models import Staff

from staff.person.models import (
    Contact,
    ContactType,
)


def _get_contact_type():
    return ContactType.objects.get(id=9)


def _get_contacts_dict(persons):
    contacts = (
        Contact.objects
        .filter(person_id__in=[person['id'] for person in persons])
        .order_by('person', 'position')
    )
    return {
        person_id: list(contacts)
        for person_id, contacts
        in groupby(contacts, lambda contact: contact.person.id)
    }


def _email_in_contacts(email, contacts, contact_type_id):
    for contact in contacts:
        if (contact.contact_type_id == contact_type_id
                and contact.account_id == email):
            return True
    return False


def _insert_email(person_id, email, contacts, contact_type_id):
    contacts.insert(0, Contact(
        person_id=person_id,
        contact_type_id=contact_type_id,
        account_id=email,
    ))
    for position, contact in enumerate(contacts, start=1):
        contact.position = position
        contact.save()


def migrate_wrong_domain_emails():
    contact_type = _get_contact_type()
    contact_type_id = contact_type.id

    persons = (
        Staff.objects
        .values('id', 'work_email')
        .exclude(login=F('login_mail'))
    )

    all_contacts = _get_contacts_dict(persons)

    for person in persons:
        email = person['work_email']
        if email:
            contacts = all_contacts.get(person['id'], [])
            if not _email_in_contacts(email, contacts, contact_type_id):
                _insert_email(person['id'], email, contacts, contact_type_id)
