from itertools import groupby

from staff.person.models import Staff, Contact
from staff.person.controllers import PersonCtl


def _get_contacts_dict(persons):
    contacts = (
        Contact.objects
        .filter(person_id__in=[person.id for person in persons])
        .order_by('person', 'position')
    )
    return {
        person_id: list(contacts)
        for person_id, contacts
        in groupby(contacts, lambda contact: contact.person.id)
    }


def _email_in_contacts(email, contacts, contact_type_id):
    for contact in contacts:
        if (contact.contact_type_id == contact_type_id
                and contact.account_id == email):
            return True
    return False


def _insert_email(person_id, email, contacts, contact_type_id):
    contacts.insert(0, Contact(
        person_id=person_id,
        contact_type_id=contact_type_id,
        account_id=email,
    ))
    for position, contact in enumerate(contacts, start=1):
        contact.position = position
        contact.save()


def migrate_login_passports():
    contact_type_id = 1  # Личная почта

    all_persons = (
        Staff.objects
        .filter(is_login_passport_confirmed=0)
        .exclude(login_passport=None)
    )
    total = all_persons.count()
    print('Persons to do: %s' % total)

    all_contacts = _get_contacts_dict(all_persons)
    n = 0
    for person in all_persons:
        email = person.login_passport + '@yandex.ru'

        contacts = all_contacts.get(person.id, [])
        if not _email_in_contacts(email, contacts, contact_type_id):
            _insert_email(person.id, email, contacts, contact_type_id)

        PersonCtl(person).update({'login_passport': None}).save(person.user)
        n += 1
        if not n % 100:
            print('%s from %s done...' % (n, total))
