import csv

from datetime import datetime

from django.core.management.base import BaseCommand
from django.db.models import Max

from phonenumbers import parse, format_number, PhoneNumberFormat

from staff.lib.db import atomic
from staff.person.models import Staff, StaffPhone, PHONE_KIND, PHONE_PROTOCOLS, PHONE_TYPES


class Command(BaseCommand):
    args = 'csv_file'
    help = 'Imports csv file with staff logins and mobile phones'

    def handle(self, *args, **options):
        rows = self._read_rows_from_csv(args[0])
        found_persons = self._load_persons_from_db(rows.keys())
        max_pos_by_person = self._max_position_by_login(found_persons)

        with atomic():
            for login, max_pos in max_pos_by_person:
                person_id = found_persons[login]
                phone_number = rows[login]
                position = max_pos + 1 if max_pos else 0

                self.stdout.write('Creating phone record for {}. '.format(login), ending='')
                self._add_phone(person_id, phone_number, position)
                self.stdout.write('Done')

    def _check_for_missed_logins(self, found_persons, logins):
        found_persons = set(found_persons)
        for login in logins:
            if login not in found_persons:
                msg = 'Login {person} doesn\'t exist. Skip it'.format(person=login)
                self.stderr.write(msg)

    def _read_rows_from_csv(self, file_path):
        self.stdout.write('Opening {}'.format(file_path))

        with open(file_path, 'rU') as csv_file:
            csv_reader = csv.reader(csv_file, delimiter=str(';').encode('utf-8'))
            return {staff_login.strip(): phone for staff_login, phone in csv_reader}

    def _load_persons_from_db(self, logins):
        self.stdout.write('Searching for staff logins')
        found_persons = dict(
            Staff.objects
            .filter(login__in=logins)
            .values_list('login', 'id')
        )

        self._check_for_missed_logins(found_persons.keys(), logins)

        return found_persons

    def _max_position_by_login(self, found_persons):
        return (
            Staff.objects
            .filter(login__in=found_persons.keys())
            .annotate(max_pos=Max('phones__position'))
            .values_list('login', 'max_pos')
        )

    def _add_phone(self, person_id, phone_number, position):
        phone_number = format_number(parse(phone_number, 'RU'),  PhoneNumberFormat.INTERNATIONAL)
        now = datetime.now()

        StaffPhone.objects.create(
            staff_id=person_id,
            number=phone_number,
            type=PHONE_TYPES.MOBILE,
            protocol=PHONE_PROTOCOLS.ALL,
            kind=PHONE_KIND.COMMON,
            position=position,
            modified_at=now,
            created_at=now,
        )
