import csv

from django.core.management.base import BaseCommand
from django.db.models import Max

from staff.lib.db import atomic
from staff.person.models import Staff, Contact, ContactType


class Command(BaseCommand):
    args = 'csv_file'
    help = 'Imports csv file with staff logins and telegram logins'

    def handle(self, *args, **options):
        rows = self._read_rows_from_csv(args[0])
        contact_type = self._find_telegram_contact_type()
        found_persons = self._load_persons_from_db(rows.keys())
        found_persons = self._without_persons_that_already_have_telegram(found_persons, contact_type)
        max_pos_by_person = self._max_position_by_login(found_persons)

        with atomic():
            for login, max_pos in max_pos_by_person:
                person_id = found_persons[login]
                telegram_login = rows[login]
                position = max_pos + 1 if max_pos else 0

                self.stdout.write('Creating contact record for {}. '.format(login), ending='')
                self._create_contact(person_id, contact_type, telegram_login, position)
                self.stdout.write('Done')

    def _check_for_missed_logins(self, found_persons, logins):
        found_persons = set(found_persons)
        for login in logins:
            if login not in found_persons:
                msg = 'Login {person} doesn\'t exist. Skip it'.format(person=login)
                self.stderr.write(msg)

    def _find_telegram_contact_type(self):
        self.stdout.write('Searching for telegram contact type')
        return ContactType.objects.get(name__iexact='telegram')

    def _read_rows_from_csv(self, file_path):
        self.stdout.write('Opening {}'.format(file_path))

        with open(file_path, 'rU') as csv_file:
            csv_reader = csv.reader(csv_file, delimiter=str(';'))
            return {staff_login.strip(): telegram for staff_login, telegram in csv_reader}

    def _load_persons_from_db(self, logins):
        self.stdout.write('Searching for staff logins')
        found_persons = dict(
            Staff.objects
            .filter(login__in=logins)
            .values_list('login', 'id')
        )

        self._check_for_missed_logins(found_persons.keys(), logins)

        return found_persons

    def _without_persons_that_already_have_telegram(self, found_persons, contact_type):
        persons_with_telegram = set(
            Contact.objects
            .filter(person__login__in=found_persons.keys())
            .filter(contact_type=contact_type)
            .values_list('person__login', flat=True)
        )

        return {
            login: found_persons[login]
            for login in found_persons
            if login not in persons_with_telegram
        }

    def _max_position_by_login(self, found_persons):
        return (
            Staff.objects
            .filter(login__in=found_persons.keys())
            .annotate(max_pos=Max('contact__position'))
            .values_list('login', 'max_pos')
        )

    def _create_contact(self, person_id, contact_type, telegram_login, position):
        Contact.objects.create(
            person_id=person_id,
            contact_type=contact_type,
            private=False,
            account_id=telegram_login,
            position=position
        )
