import os
import csv

from django.core.management.base import BaseCommand, CommandError
from django.db.models import Q

from staff.person_profile.controllers import photos as ctl
from staff.preprofile.utils import normalize_phone

from staff.person.models import Staff, StaffExtraFields, Contact, ContactType


class Command(BaseCommand):
    help = '!!! HAVE NOT BEEN TESTED AS A COMMAND !!! Uploads dismissed users from a csv file'

    def add_arguments(self, parser):
        parser.add_argument("file_name", nargs=1, type=str, help="CSV file to load users from")
        parser.add_argument("min_matches", nargs='?', type=int, default=3, help="minimum matches to update data")
        parser.add_argument("run_db_update", nargs='?', type=bool, default=False, help="update data in models")
        parser.add_argument("and_this_time_i_mean_it", nargs='?', type=bool, default=False, help="save to DB")

    def handle(self, *args, **options):
        try:
            upload_from_csv(**options)
        except Exception as e:
            raise CommandError(e)


def upload_from_csv(file_name, min_matches=3, **kwargs):
    mapping = {
        "last_name": 0,
        "first_name": 1,
        "middle_name": 2,
        "login": 5,
        "login_passport": 6,
        "mobile_phone": 7,
        "home_email": 9,
        "login_skype": 11,
        "home_page": 12,
    }

    old_names_column = 4
    telegram_column = 10
    telegram_contact_type = ContactType.objects.get(name="Telegram")

    wipe_fields = [
        # все мобильные
        "home_phone", "mobile_phone", "phones",

        # аккаунты
        "login_lj", "login_mail", "login_mk", "login_skype", "login_twitter",

        # адрес
        "address", "address_en",

        # образование, учебное заведение
        "edu_direction", "edu_place", "edu_place_en", "edu_status",
    ]

    with open(file_name, "r") as f:
        reader = csv.reader(f, delimiter=';')
        out_file_name = os.path.dirname(file_name) + "out-" + os.path.basename(file_name)

        with open(out_file_name, "w") as rf:
            writer = csv.writer(rf, delimiter=';')

            for row in reader:
                user_data = dict()

                for field in mapping.keys():
                    value = row[mapping[field]].strip()

                    if field == "mobile_phone":
                        value = normalize_phone(value)
                        if value is None:
                            value = ""

                    if field == "login":
                        value = value.lower()

                    if field in ["login", "login_passport"]:
                        value = parse_login(value)

                    user_data[field] = [value, value]

                new_full_name = row[old_names_column].strip().split()
                if len(new_full_name) == 3:
                    user_data["first_name"][0] = new_full_name[1]
                    user_data["middle_name"][0] = new_full_name[2]
                    user_data["last_name"][0] = new_full_name[0]

                new_telegram = row[telegram_column].strip()
                if new_telegram.startswith("https://t.me/"):
                    new_telegram = new_telegram[len("https://t.me/"):]
                if new_telegram.startswith("@"):
                    new_telegram = new_telegram[1:]
                if not any(x.isalpha() for x in new_telegram):
                    new_telegram = ""   # ignore phone numbers
                if " " in new_telegram:
                    new_telegram = ""  # ignore flights of the thought
                telegram = []

                if new_telegram:
                    telegram = (
                        Contact.objects
                        .filter(contact_type__name="Telegram")
                        .filter(Q(account_id=new_telegram) | Q(account_id="@" + new_telegram))
                        .values()
                    )

                user_filter = Q()

                for field in mapping.keys():
                    if user_data[field][0]:
                        user_filter |= Q(**{field: user_data[field][0]})

                found = Staff.objects.filter(is_robot=False, is_dismissed=True).filter(user_filter)
                best_match = 0
                candidates = list()

                for usr in found:
                    matched_columns = list()

                    for field in mapping.keys():
                        if user_data[field][0] and (getattr(usr, field) == user_data[field][0]):
                            matched_columns.append(field)

                    match = len(matched_columns)
                    telegram_match = usr.id in [x["person_id"] for x in telegram]

                    if telegram_match:
                        match += 1

                    if match > best_match:
                        best_match = match
                        candidates.clear()

                    if match == best_match:
                        candidates.append((usr, matched_columns, telegram_match))

                result = f"Found {len(candidates)} candidates with {best_match} "
                result += f"matched: {','.join([str((c[0].id, c[0].login, c[1], c[2])) for c in candidates])} "

                if best_match >= min_matches and len(candidates) == 1:
                    candidate, matched_columns, matched_telegram = candidates[0]
                    result = f"Updating {candidate.id, candidate.login} "
                    result += f"({best_match} matched columns: {matched_columns}, telegram: {matched_telegram})"

                    not_updated = set([x for x in mapping.keys() if x not in matched_columns if x != "login"])
                    not_matched = not_updated.union(["last_name", "first_name", "middle_name"])
                    columns_to_update = [x for x in not_matched if (getattr(candidate, x) or "") != user_data[x][1]]

                    result += ", patching "
                    result += str([f"{x}: {getattr(candidate, x)} -> {user_data[x][1]}" for x in columns_to_update])

                    if kwargs.get("run_db_update", False):
                        for field in wipe_fields:
                            result += f", -{field} '{getattr(candidate, field)}'"
                            setattr(candidate, field, "")

                        for field in candidate.contact_set.all():
                            result += f", -contact '{field.account_id}' ({field.contact_type.name})"

                        for field in candidate.phones.all():
                            result += f", -phone {field.number} ({field.type}, '{field.description}')"

                        contact = None
                        if new_telegram:
                            result += f", +contact '{new_telegram}' ({telegram_contact_type.name})"
                            contact = Contact(
                                account_id=new_telegram,
                                contact_type=telegram_contact_type,
                                person_id=candidate.id,
                                private=True,
                            )

                        try:
                            extra = candidate.extra
                        except StaffExtraFields.DoesNotExist:
                            extra = StaffExtraFields(staff=candidate, staff_agreement=False)
                            pass

                        if not extra.staff_agreement:
                            result = "*** NO AGREEMENT *** " + result
                            extra.staff_agreement = True

                        for field in columns_to_update:
                            setattr(candidate, field, user_data[field][1])

                        try:
                            if kwargs.get("and_this_time_i_mean_it", False):
                                for x in candidate.contact_set.all():
                                    x.delete()
                                for x in candidate.phones.all():
                                    x.delete()
                                extra.save()
                                candidate.save()
                                if contact:
                                    contact.save()

                                ctl.PhotosCtl(candidate).delete_all_photos()
                        except Exception as ex:
                            result = "FAILED: " + str(ex) + result

                print(result)
                row.append(result)
                writer.writerow(row)


def parse_login(login_str: str) -> str or None:
    if login_str.startswith("@"):
        login_str = login_str[1:]

    if '@' in login_str:
        login_str = login_str[:login_str.index('@')]

    if not login_str:
        return None

    return login_str
