import functools

from staff.person.models import Staff

from django.core.exceptions import ObjectDoesNotExist


def _true_or_raise_method(exception_class):
    def decorator(meth):
        @functools.wraps(meth)
        def decorated_method(self, *args, **kwargs):
            if self:
                return meth(self, *args, **kwargs)
            else:
                raise exception_class
        return decorated_method
    return decorator


class Memorial(object):

    class NotMemorialError(Exception):
        """Для этого сотрудника не задан мемориальный профиль"""

    def __init__(self, person):
        self.person = person

    @_true_or_raise_method(NotMemorialError)
    def get_birth_date(self):
        return self.person.birthday

    @_true_or_raise_method(NotMemorialError)
    def get_death_date(self):
        return self.person.memorial_profile.death_date

    @_true_or_raise_method(NotMemorialError)
    def get_status_text_unclean(self):
        return self.person.memorial_profile.i_status_block_html

    @_true_or_raise_method(NotMemorialError)
    def get_status_text(self):
        return (self.get_status_text_unclean()
                    .replace('\'', '\\')
                    .replace('"', '\\"')
                    .replace('\n', '\\n')
                    .replace('\r', ''))

    def _get_context(self):
        if self:
            return {
                'year_birth': self.get_birth_date().year,
                'year_death': self.get_death_date().year,
                'status_text': self.get_status_text(),
            }
        else:
            return {}

    def __getitem__(self, item):
        return self._get_context()[item]

    def __bool__(self):
        try:
            return self.person.memorial_profile.intranet_status > 0
        except (ObjectDoesNotExist, AttributeError):
            return False

    @classmethod
    def get_many(cls, person_logins):
        return [cls(person) for person
                in Staff.objects.select_related('memorial_profile')
                                .filter(login__in=person_logins)]
