import logging
import re
from staff.lib import requests
from hashlib import md5
import yenv

from django.conf import settings
from django.core.cache import caches
from django.utils.encoding import smart_str


INFLECTOR_SCHEME = 'http'
INFLECTOR_PORT = '8891'
INFLECTOR_HOST = 'hamzard.yandex.net'
if yenv.type.lower() == 'production':
    INFLECTOR_HOST = 'reqwizard.yandex.net'

INFLECTOR_WIZARD_URL = '%s://%s:%s/wizard' % (
    INFLECTOR_SCHEME,
    INFLECTOR_HOST,
    INFLECTOR_PORT,
)
INFLECTOR_WIZARD_TIMEOUT = 2

CACHE_TIME = 60 * 60 * 24 * 365  # 1 год
cache = caches[getattr(settings, 'INFLECTOR_CACHE', 'default')]

logger = logging.getLogger(__name__)


class InflectorException(Exception):
    """Базовое исключение инфлектора"""


class InflectorRequestError(InflectorException):
    """Ошибка выполнениея запроса к инфлектору"""


class InflectorParseError(InflectorException):
    """Ошибка разбора ответа инфлектора"""


msg = 'Could not inflect word "%s" because the following error was occurred.'


def wizard_request(word, timeout=None):
    """
    Выполняет запрос к инфлектору.

    @param word: значение склоняемого слова или словосочетания
    @param timeout: время в секундах, сколько ждем ответа

    @see: https://wiki.yandex-team.ru/poiskovajaplatforma/lingvistika/wizard/wizardasservis/inflector/
    """

    if timeout is None:
        timeout = INFLECTOR_WIZARD_TIMEOUT

    params = {'action': 'inflector', 'text': word}

    try:
        response = requests.get(
            INFLECTOR_WIZARD_URL,
            params=params,
            timeout=timeout
        )
    except requests.ConnectionError:
        logger.exception(msg, word)
        raise InflectorRequestError

    try:
        response.raise_for_status()
    except requests.HTTPError:
        logger.exception(msg, word)
        raise InflectorRequestError

    try:
        data = response.json()
    except ValueError:
        logger.exception(msg, word)
        raise InflectorParseError

    try:
        data = [v['Text'] for v in data['Form']]
    except KeyError:
        logger.exception(msg, word)
        raise InflectorParseError

    return data


def get_inflect_form(word, form_id):
    # делаем сдвиг в индексе,
    # так как в ответе приезжает список с нулевым начальным индексом
    try:
        return wizard_request(word)[form_id - 1]
    except IndexError:
        logger.exception(msg, word)
        raise InflectorParseError


def build_cache_key(word, form_id):
    return 'inflect_v2:%s:%s' % (md5(smart_str(word).encode('utf-8')).hexdigest(), form_id)


def inflect(word, form_id, use_cache=True):
    if use_cache:
        cache_key = build_cache_key(word, form_id)
        result = cache.get(cache_key)
        if result is not None:
            return result

    try:
        word = get_inflect_form(word, form_id)
    except InflectorException:
        pass
    else:
        if use_cache:
            cache.set(cache_key, word, CACHE_TIME)  # 1 год

    return word


pattern = '%s{%s}'
attr_key_map = (
    ('first_name', 'persn'),
    ('last_name', 'famn'),
    ('surname', 'patrn'),
    ('gender', 'gender'),
)
attr_re_list = tuple(
    re.compile(r'%s\{(?P<%s>.*?)\}' % (key, attr))
    for attr, key in attr_key_map
)


def build_word_from_fio(fio):
    word_list = []
    for atrr, key in attr_key_map:
        value = fio[atrr]
        if value:
            word_list.append(pattern % (key, value))
    word = ''.join(word_list)
    word += ';fio=1'
    return word


def extract_fio_from_word(word):
    fio = {}
    for attr_re in attr_re_list:
        match = attr_re.search(word)
        if match:
            fio.update(match.groupdict())
    return fio


def inflect_fio(first_name, last_name, form_id, gender=None, surname=None):
    """
    Получить склонение ФИО в заданной форме.

    @param first_name: имя
    @param last_name: фамилия
    @param form_id: индекс требуемого падежа, начиная с единицы
    @param gender: пол, мужской или женский. m или f соответственно.
    @param surname: отчество
    """
    gender = gender and gender.lower()
    if gender not in (None, 'm', 'f'):
        raise RuntimeError('Gender must be m or f. "%s" received.' % gender)

    fio = {
        'first_name': first_name,
        'last_name': last_name,
        'gender': gender,
        'surname': surname,
    }

    word = build_word_from_fio(fio)
    word = inflect(word, form_id)

    fio.update(extract_fio_from_word(word))

    if surname:
        result = '%(first_name)s %(surname)s %(last_name)s' % fio
    else:
        result = '%(first_name)s %(last_name)s' % fio

    return result
