from django.db import models

from django.utils.translation import ugettext_lazy as _

from staff.lib.utils.enum_choices import EnumChoices
from staff.lib.utils.ordered_choices import OrderedChoices
from staff.lib.models.base import IntranetModel, SortedModel, AtomicSaveModel


PHONE_TYPES = OrderedChoices(
    ('MOBILE', 2, _('intranet_stuff.StaffPhones_Mobile')),
    ('HOME', 3, _('intranet_stuff.StaffPhones_Home')),
)

PHONE_KIND = OrderedChoices(
    ('COMMON', 'common'),
    ('EMERGENCY', 'emergency'),
    ('MONITORING', 'monitoring'),
    ('HIDDEN', 'hidden'),
)

PHONE_PROTOCOLS = OrderedChoices(
    ('ALL', 'all'),
    ('SMS', 'sms'),
    ('VOICE', 'voice'),
)

VALIDATION_TYPES = OrderedChoices(
    ('LOGIN', 'login'),
    ('EMAIL', 'email'),
    ('URL', 'url'),
)

VERIFY_STATE = OrderedChoices(
    ('WAIT', 'wait'),
    ('VERIFIED', 'verified'),
    ('FAILED', 'failed'),
    ('CONFIRMED', 'confirmed'),
)

VERIFY_CODE_OBJECT_TYPES = OrderedChoices(
    ('STAFF', 'staff'),
    ('OFFER', 'offer'),
    ('PREPROFILE', 'preprofile'),
)


class ContactType(SortedModel):
    name = models.CharField(max_length=32)
    name_en = models.CharField(max_length=32, default='')
    icon_url = models.CharField(max_length=256, default='')
    url_pattern = models.CharField(max_length=128, default='')
    validation_type = models.CharField(
        max_length=10,
        choices=VALIDATION_TYPES,
    )

    show_in_vcard = models.BooleanField(default=False)
    vcard_template = models.TextField(default='', blank=True)

    def __str__(self):
        return self.name


class ContactTypeId(EnumChoices):
    telegram = 18
    another_work_email = 9
    assistant = 20


class Contact(SortedModel, AtomicSaveModel):
    person = models.ForeignKey('django_intranet_stuff.Staff')
    contact_type = models.ForeignKey(ContactType)
    account_id = models.CharField(max_length=100)
    private = models.BooleanField(default=False)

    def __str__(self):
        return '{contact_type}:{contact_id}:{private}'.format(
            contact_type=self.contact_type,
            contact_id=self.account_id,
            private='private' if self.private else 'public',
        )


class StaffPhone(IntranetModel, SortedModel):
    staff = models.ForeignKey('Staff', related_name='phones')
    number = models.CharField(max_length=50)
    type = models.PositiveSmallIntegerField(choices=PHONE_TYPES.choices(), default=PHONE_TYPES.MOBILE)
    protocol = models.CharField(max_length=50, choices=PHONE_PROTOCOLS.choices(), default=PHONE_PROTOCOLS.ALL)
    kind = models.CharField(max_length=50, choices=PHONE_KIND.choices(), default=PHONE_KIND.COMMON)
    description = models.CharField(max_length=255, default='')
    for_digital_sign = models.BooleanField(default=False)

    def __str__(self):
        return 'Phone number %s of %s' % (self.number, self.staff)

    class Meta(IntranetModel.Meta):
        db_table = 'intranet_staff_phone'


class VerifyCode(models.Model):
    object_type = models.CharField(
        max_length=16,
        choices=VERIFY_CODE_OBJECT_TYPES.choices(),
        default=VERIFY_CODE_OBJECT_TYPES.STAFF,
        null=True,
    )
    object_id = models.IntegerField(null=True)
    phone_number = models.CharField(max_length=255, null=True)
    code = models.CharField(max_length=20)
    state = models.CharField(
        max_length=10,
        choices=VERIFY_STATE.choices(),
        default=VERIFY_STATE.WAIT,
    )
    tries = models.IntegerField(default=0)

    phone = models.ForeignKey(
        to=StaffPhone,
        related_name='codes',
        null=True,
        blank=True,
    )
    person = models.ForeignKey(
        to='Staff',
        related_name='codes',
        null=True,
        blank=True,
    )

    created_at = models.DateTimeField(auto_now_add=True)
    modified_at = models.DateTimeField(auto_now=True)

    def attach_to_staff(self, phone: StaffPhone, commit=True):
        self.phone = phone
        self.person = phone.staff
        if commit:
            self.save()

    def __str__(self):
        return f'VerifyCode for object_id {self.object_id} of type {self.object_type}'

    class Meta(IntranetModel.Meta):
        db_table = 'intranet_verify_code'
