from django.db import models

from staff.lib.utils.ordered_choices import OrderedChoices
from staff.lib.models.base import I18nModel, AtomicSaveModel

from staff.person_avatar.models import AvatarMetadata
from staff.person.models import PaymentSystem
from staff.person.models.occupation import Occupation

RESPONSIBLE_ROLES = OrderedChoices(
    ('OWNER', 'owner'),
    ('USER', 'user'),
)


class StaffExtraFields(AtomicSaveModel):
    staff = models.OneToOneField(
        'django_intranet_stuff.Staff',
        related_name='extra',
    )
    nda_end_date = models.DateField(null=True)
    contract_end_date = models.DateField(null=True)
    byod_access = models.BooleanField(default=False)
    wiretap = models.BooleanField(default=False)
    staff_agreement = models.BooleanField(default=False)
    staff_biometric_agreement = models.BooleanField(default=False)

    oebs_first_name = models.CharField(
        blank=True,
        max_length=255,
        default='',
        verbose_name="Имя сотрудника",
    )
    oebs_middle_name = models.CharField(
        blank=True,
        max_length=255,
        default='',
        verbose_name="Отчество сотрудника",
    )
    oebs_last_name = models.CharField(
        blank=True,
        max_length=255,
        default='',
        verbose_name="Фамилия сотрудника",
    )
    oebs_address = models.CharField(
        blank=True,
        max_length=255,
        default='',
        verbose_name="Адрес проживания",
    )
    oebs_manage_org_name = models.CharField(
        blank=True,
        max_length=255,
        default='',
        verbose_name="Кадровое наименование подразделения",
    )
    oebs_headcount = models.BooleanField(
        default=False,
    )

    oebs_payment_system = models.ForeignKey(
        to=PaymentSystem,
        null=True,
        default=None,
    )

    main_img_for_upload = models.ForeignKey(
        AvatarMetadata,
        related_name='main_img',
        null=True
    )
    avatar_img_for_upload = models.ForeignKey(
        AvatarMetadata,
        related_name='avatar_img',
        null=True
    )
    gravatar_img_for_upload = models.ForeignKey(
        AvatarMetadata,
        related_name='gravarar_img',
        null=True
    )
    date_survey_letter = models.DateField(null=True)
    is_welcome_mail_sent = models.BooleanField(default=False)
    is_offer_avatar_uploaded = models.BooleanField(default=False)
    last_vacation_accrual_at = models.DateField(null=True, auto_now_add=True)
    occupation = models.ForeignKey(Occupation, null=True)


class WelcomeEmailTemplate(models.Model):
    text = models.TextField(blank=True)
    text_en = models.TextField(blank=True)


class PersonBlockActions(models.Model):
    person = models.ForeignKey('django_intranet_stuff.Staff')
    is_passport_blocked = models.BooleanField(default=False)
    passport_block_last_action_timestamp = models.DateTimeField(auto_now=True)
    passport_block_last_action_reason = models.TextField(blank=True, null=True)
    created_at = models.DateTimeField(auto_now_add=True)

    def __str__(self):
        return '{person}: is_passport_blocked={is_passport_blocked}'.format(
            person=self.person.login,
            is_passport_blocked=self.is_passport_blocked,
        )


class ResponsibleForRobot(AtomicSaveModel):
    responsible = models.ForeignKey('django_intranet_stuff.Staff', related_name='robots')
    robot = models.ForeignKey('django_intranet_stuff.Staff', related_name='responsibles')
    role = models.CharField(max_length=10, choices=RESPONSIBLE_ROLES)


class MemorialProfile(I18nModel, AtomicSaveModel):
    person = models.OneToOneField(
        'django_intranet_stuff.Staff', related_name='memorial_profile', verbose_name='Сотрудник'
    )
    intranet_status = models.IntegerField(
        default=0, verbose_name='Признак активности', help_text='0=Выкл, 1=Вкл'
    )
    status_block_html = models.TextField(verbose_name='Код содержимого плашки', default='')
    status_block_html_en = models.TextField(verbose_name='Код содержимого плашки (англ)', default='')
    death_date = models.DateField(verbose_name='Дата смерти')

    def __str__(self):
        return '{self.person.login}'.format(self=self)

    class Meta:
        db_table = 'intranet_memorial_profile'
        app_label = 'django_intranet_stuff'
