from typing import Set

from django.db.models import Q, QuerySet
from django.db import models
from django.utils.translation import ugettext_lazy as _

from staff.lib.models.mptt import get_heirarchy_filter_query
from staff.lib.utils.ordered_choices import OrderedChoices

from staff.lib.models.base import IntranetModel, IntranetManager
from staff.lib.exceptions import SearchError


DOMAIN = OrderedChoices(
    ('AUTO_RU', 'A', 'auto.ru'),
    ('COMPTEK_RU', 'C', 'comptek.ru'),
    ('YANDEX_TEAM_RU', 'Y', 'yandex-team.ru'),
    ('YANDEX_TEAM_COM_UA', 'U', 'yandex-team.com.ua'),
    ('YAMONEY_RU', 'M', 'yamoney.ru'),
    ('YAPROBKI_RU', 'P', 'yaprobki.ru'),
    ('YANDEX_TEAM_COM', 'W', 'yandex-team.com'),
    ('YANDEX_TEAM_COM_TR', 'T', 'yandex-team.com.tr'),
)

GENDER = OrderedChoices(
    ('MALE', 'M', _('intranet_stuff.Staff_Male')),
    ('FEMALE', 'F', _('intranet_stuff.Staff_Female')),
)
FAMILY_STATUS = OrderedChoices(
    ('SINGLE', 'S', _('intranet_stuff.Staff_SingleMale')),
    ('MARRIED', 'M', _('intranet_stuff.Staff_MarriedMale')),
)
EDU_STATUS = OrderedChoices(
    ('SECONDARY', 'S', _('intranet_stuff.Staff_SecondaryEducation')),
    ('INCOMPLETE', 'I', _('intranet_stuff.Staff_HigherIncompleteEducation')),
    ('BACHELOR', 'B', _('intranet_stuff.Staff_BachelorsDegreeEducation')),
    ('MASTER', 'M', _('intranet_stuff.Staff_MastersDegreeEducation')),
    ('SPECIALIST', 'P', _('intranet_stuff.Staff_SpecialistDegreeEducation')),
    ('ACADEMIC', 'D', _('intranet_stuff.Staff_AcademicDegreeEducation')),
)
EDU_DIRECTION = OrderedChoices(
    ('TECHNICAL', 'T', _('intranet_stuff.Staff_TechnicalDirection')),
    ('LIBERAL', 'L', _('intranet_stuff.Staff_LiberalDirection')),
    ('NATURAL', 'N', _('intranet_stuff.Staff_NaturalDirection')),
    ('ECONOMIC', 'E', _('intranet_stuff.Staff_EconomicDirection')),
    ('BUSINESS', 'B', _('intranet_stuff.Staff_BusinessDirection')),
)
EMPLOYMENT = OrderedChoices(
    ('FULL', 'F', _('intranet_stuff.Staff_FullEmployment')),
    ('PARTIAL', 'P', _('intranet_stuff.Staff_PartTimeEmployment')),
    ('SECONDARY', 'D', _('intranet_stuff.Staff_SecondaryJob')),
)
LANG = OrderedChoices(
    ('EN', 'en', 'English'),
    ('RU', 'ru', 'Russian'),
    ('TR', 'tr', 'Turkish'),
)
TSHIRT_SIZE = OrderedChoices(
    ('XXS', 'XXS', 'XXS'),
    ('XS', 'XS', 'XS'),
    ('S', 'S', 'S'),
    ('M', 'M', 'M'),
    ('L', 'L', 'L'),
    ('XL', 'XL', 'XL'),
    ('XXL', 'XXL', 'XXL'),
    ('XXXL', 'XXXL', 'XXXL'),
    ('4XL', '4XL', '4XL'),
)


AFFILIATION = OrderedChoices(
    ('YANDEX', 'yandex', _('intranet_stuff.Staff_AffiliationYandex')),
    ('YAMONEY', 'yamoney', _('intranet_stuff.Staff_AffiliationYamoney')),
    ('EXTERNAL', 'external', _('intranet_stuff.Staff_AffiliationExternal')),
)


WORK_MODES = OrderedChoices(
    ('REMOTE', 'remote', _('intranet_stuff.Staff_WorkModeRemote')),
    ('OFFICE', 'office', _('intranet_stuff.Staff_WorkModeOffice')),
    ('MIXED', 'mixed', _('intranet_stuff.Staff_WorkModeMixed')),
)


class UserNotFound(SearchError):
    code = 'no_such'


class UserFoundMultiple(SearchError):
    code = 'many'


class StaffManager(IntranetManager):
    def get_by_user(self, user):
        if user and user.is_authenticated():
            try:
                return self.get(login=user.username)
            except Staff.DoesNotExist:
                pass
        return None

    def search_exactly(self, query, _fields=('is_dismissed', )):
        """
        Search exact fit user by query (login, name or surname)
        Used in interfaces as defense of user's autocomlete from incorrect input
        """
        if 'is_dismissed' in _fields:
            qs = self.filter(is_dismissed=False)
        else:
            qs = self.all()
        try:
            if len(query.split(' ')) == 2:
                t = query.split(' ')
                try:
                    r = qs.get(first_name=t[0], last_name=t[1])
                except Staff.DoesNotExist:
                    try:
                        r = qs.get(first_name=t[1], last_name=t[0])
                    except Staff.DoesNotExist:
                        raise UserNotFound('User "%s" has not been found' % query)
            else:
                try:
                    r = qs.get(login=query)
                except Staff.DoesNotExist:
                    try:
                        r = qs.get(last_name=query)
                    except Staff.DoesNotExist:
                        try:
                            r = qs.get(first_name=query)
                        except Staff.DoesNotExist:
                            raise UserNotFound('User "%s" has not been found' % query)
        except Staff.MultipleObjectsReturned:
            raise UserFoundMultiple('There are several users named "%s"' % query)
        return r


class Staff(IntranetModel):

    user = models.OneToOneField('users.User', null=True)
    from_staff_id = models.PositiveIntegerField(db_index=True, default=0)
    # В терминах blackbox и passport это поле называется display_name.
    # Оно содержит ненормализованный логин, каким его видит пользователь, с точками.
    login = models.CharField(max_length=50, db_index=True, default='')
    # Нормализованный логин, который не содержит точек.
    normal_login = models.CharField(max_length=50, db_index=True, default='')

    login_mail = models.CharField(max_length=50, db_index=True, default='')
    domain = models.CharField(max_length=1, choices=DOMAIN.choices(), default='')
    login_ld = models.CharField(max_length=50, db_index=True, unique=True)

    uid = models.CharField(max_length=16, db_index=True, null=True, unique=True)
    guid = models.CharField(max_length=47, null=True, default='')

    login_passport = models.CharField(max_length=100, db_index=True, null=True, unique=True)

    first_name = models.CharField(max_length=50, default='', db_index=True)
    first_name_en = models.CharField(max_length=50, default='', db_index=True)
    middle_name = models.CharField(max_length=50, default='')
    hide_middle_name = models.BooleanField(default=True)
    preferred_name = models.CharField(max_length=50, default='')
    preferred_name_en = models.CharField(max_length=50, default='')
    last_name = models.CharField(max_length=100, default='', db_index=True)
    last_name_en = models.CharField(max_length=100, default='', db_index=True)
    en_name = models.CharField(max_length=255, default='')
    tz = models.CharField(max_length=30, default='')
    auto_translate = models.BooleanField(default=False)
    date_completion_internship = models.DateField(null=True, default=None)

    # https://st.yandex-team.ru/STAFF-3520
    is_homeworker = models.BooleanField(default=False)
    is_robot = models.BooleanField(default=False)
    affiliation = models.CharField(max_length=32, db_index=True, choices=AFFILIATION.choices())

    work_mode = models.CharField(max_length=16, choices=WORK_MODES.choices(), default=None, null=True, blank=True)

    # https://st.yandex-team.ru/ML-1291
    has_exchange = models.BooleanField(default=False)

    has_namesake = models.BooleanField(default=False)

    birthday = models.DateField(null=True)
    gender = models.CharField(max_length=1, choices=GENDER.choices(), default='')

    family_status = models.CharField(max_length=1, choices=FAMILY_STATUS.choices(), default='')
    children = models.PositiveSmallIntegerField(null=True)

    car = models.CharField(max_length=100, default='')  # TODO: удалить
    car_num = models.CharField(max_length=100, default='')  # TODO: удалить

    address = models.CharField(max_length=1024, default='')
    address_en = models.CharField(max_length=1024, default='')

    edu_status = models.CharField(max_length=1, choices=EDU_STATUS.choices(), default='')
    edu_direction = models.CharField(max_length=1, choices=EDU_DIRECTION.choices(), default='')
    edu_place = models.CharField(max_length=255, default='')
    edu_place_en = models.CharField(max_length=255, default='')
    edu_date = models.DateField(null=True)

    department = models.ForeignKey('Department', null=True, on_delete=models.SET_NULL)
    budget_position = models.ForeignKey(
        'budget_position.BudgetPosition',
        on_delete=models.PROTECT,
        null=True,
    )
    is_big_boss = models.BooleanField(default=False)

    is_dismissed = models.BooleanField(default=False)

    office = models.ForeignKey('Office', on_delete=models.PROTECT)

    table = models.ForeignKey('Table', null=True, on_delete=models.SET_NULL)
    room = models.ForeignKey('Room', null=True, on_delete=models.SET_NULL)

    join_at = models.DateField(null=True)
    quit_at = models.DateField(null=True)

    position = models.CharField(max_length=150, default='')
    position_en = models.CharField(max_length=150, default='')

    organization = models.ForeignKey('Organization', null=True, on_delete=models.SET_NULL)

    employment = models.CharField(max_length=1, choices=EMPLOYMENT.choices(), default='')

    work_phone = models.PositiveIntegerField(null=True, db_index=True)
    mobile_phone = models.CharField(max_length=100, default='')

    work_email = models.CharField(max_length=100, default='', db_index=True)
    home_email = models.CharField(max_length=100, default='', db_index=True)
    home_phone = models.CharField(max_length=100, default='')

    home_page = models.CharField(max_length=100, default='')
    icq = models.CharField(max_length=50, default='')
    jabber = models.CharField(max_length=100, default='')
    login_skype = models.CharField(max_length=50, default='')
    login_mk = models.CharField(max_length=50, default='')
    login_twitter = models.CharField(max_length=50, default='')
    login_lj = models.CharField(max_length=50, default='')

    login_crm = models.CharField(max_length=50, default='')

    about = models.CharField(max_length=1024, default='')
    tshirt_size = models.CharField(max_length=4, choices=TSHIRT_SIZE.choices(), default='')

    mobile_phone_model = models.CharField(max_length=100, default='')
    computer = models.CharField(max_length=150, default='')

    location_descr = models.CharField(max_length=255, default='')
    location_descr_en = models.CharField(max_length=255, default='')
    duties = models.CharField(max_length=1024, default='')
    duties_en = models.CharField(max_length=1024, default='')

    vacation = models.FloatField(null=True, default=0)
    extra_vacation = models.FloatField(null=True, default=0)
    paid_day_off = models.FloatField(null=True)

    lang_ui = models.CharField(max_length=2, choices=LANG.choices(), default='')
    lang_content = models.CharField(max_length=255, default='')

    is_login_passport_confirmed = models.BooleanField(default=False)

    show_all_middle_name = models.BooleanField(default=False)
    show_beta_interface = models.BooleanField(default=False)

    shell = models.CharField(max_length=50, default='/bin/bash')

    @property
    def is_active(self):
        return not self.is_dismissed

    @property
    def date_joined(self):
        return self.join_at

    def _get_langs(self):
        return [x for x in self.lang_content.split('|') if x]

    def _set_langs(self, langs):
        self.lang_content = '|'.join([x for x in langs if x])

    langs = property(_get_langs, _set_langs)

    @property
    def username(self):
        return self.login

    def get_full_name(self):
        name = ('%s %s' % (self.i_first_name, self.i_last_name)).strip()
        return name if name else self.login

    __str__ = get_full_name

    @property
    def departments(self):
        if self.department is None:
            return []

        result = list(self.department.get_ancestors(False))
        result.append(self.department)
        return result

    def get_superior(self):
        sups = list(self.get_superiors(depth=1, level_gte=0))
        return sups[0] if sups else None

    def get_superiors(self, depth=None, level_gte=0):
        """
        Return user's bosses ordered from small to big, up to given depth.

        If depth is None, return all bosses.

        By default level_gte is set to 0 to include boss of zero-level
        department in list (volozh).
        If it equals 1, zero-level boss will be filtered to reduce
        possible spam.
        """
        from staff.person.dis_staff_services import StaffService
        return StaffService(self).get_chiefs(depth, level_gte)

    def get_subordinates(self):
        """
        Return user's direct subordinates.
        """
        from staff.person.dis_staff_services import StaffService
        return StaffService(self).get_chief_subordinates()

    def get_all_groups(self, fields=None):

        from staff.groups.models import Group

        my_groups_qs = self.in_groups.filter(intranet_status=1)
        my_groups_qs = my_groups_qs.values('lft', 'rght', 'tree_id')

        q_gen = (Q(lft__lte=g['lft'], rght__gte=g['rght'],
                   tree_id=g['tree_id']) for g in my_groups_qs)

        q = Q()
        for elem in q_gen:
            q |= elem

        # if user isn't in any group
        if not q:
            return []

        groups_qs = (Group.objects
                     .filter(q)
                     .filter(level__gt=0, intranet_status=1))

        if fields:
            groups_qs = groups_qs.values(*fields)

        return list(groups_qs)

    @property
    def all_groups(self):
        return self.get_all_groups()

    def get_email(self):
        return self.work_email if self.work_email is not None else ''

    @property
    def email(self) -> str:
        return self.work_email

    @property
    def inflections(self):
        return StaffInflection(staff=self)

    def is_internal(self) -> bool:
        internal_employee_affiliations = (AFFILIATION.YANDEX, AFFILIATION.YAMONEY)
        return (
            self.affiliation in internal_employee_affiliations or
            self.user.has_perm('users.can_view_staff')
        )

    def has_access_to_department_profiles(self, dep_id: int) -> bool:
        if dep_id in getattr(self, '_department_profile_access_cache', {}):
            return self._department_profile_access_cache[dep_id]

        from staff.departments.models import Department

        profiles_filter_q = self.departments_by_outstaff_perm_query('django_intranet_stuff.can_view_profiles')
        departments_filter_q = self.departments_by_outstaff_perm_query('django_intranet_stuff.can_view_departments')

        assert (not profiles_filter_q and not departments_filter_q) or (profiles_filter_q and departments_filter_q)

        filter_q = Q(id=dep_id) & (profiles_filter_q | departments_filter_q)

        has_access = Department.objects.filter(filter_q).exists()
        self._department_profile_access_cache = {dep_id: has_access}
        return has_access

    def departments_by_outstaff_perm_query(self, perm: str, filter_prefix: str = '') -> Q:
        if self.is_internal():
            return Q()

        return self.departments_by_perm_query(perm, True, filter_prefix)

    def departments_by_permission(self, perm: str, instance_classes: Set[str] or None = None) -> QuerySet:
        from staff.departments.models import Department, DepartmentRole, InstanceClass
        instance_classes = instance_classes or {InstanceClass.DEPARTMENT.value}
        app_label, codename = perm.split('.')

        permission_roles_qs = (
            DepartmentRole.objects
            .filter(permissions__codename=codename, permissions__content_type__app_label=app_label)
        )

        departments_by_permission = (
            Department.all_types
            .filter(
                departmentstaff__staff=self,
                departmentstaff__role=permission_roles_qs,
                instance_class__in=instance_classes,
            )
        )

        return departments_by_permission

    def departments_by_perm_query(
        self,
        perm: str,
        by_children: bool,
        filter_prefix: str = '',
        instance_classes: Set[str] or None = None,
    ) -> Q:
        from staff.departments.models import InstanceClass
        instance_classes = instance_classes or {InstanceClass.DEPARTMENT.value}

        none_q = Q(id=None)  # Filters out all rows
        all_q = ~none_q  # Filters out nothing, we can't use Q() here as django optimizes it

        if self.user.is_superuser:
            return all_q

        departments_by_permission = list(
            self.departments_by_permission(perm, instance_classes)
            .values('lft', 'rght', 'tree_id')
        )

        if not departments_by_permission:
            return none_q

        return get_heirarchy_filter_query(
            mptt_objects=departments_by_permission,
            by_children=by_children,
            filter_prefix=filter_prefix,
            include_self=True
        )

    def save(self, **kwargs):
        """extends parent save
        Сохранить модель, записать в self.normal_login значение из self.login заменив точки на минусы
        """
        if self.login is not None:
            self.normal_login = self.login.replace('.', '-')
        else:
            self.normal_login = ""
        super(Staff, self).save()

    staff_manager = StaffManager()

    class Meta(IntranetModel.Meta):
        permissions = (
            ('can_export_staff_persons', 'Can export staff persons'),
        )
        db_table = 'intranet_staff'


class Inflection(object):

    forms = {
        'subjective': 1,
        'genitive': 2,
        'dative': 3,
        'accusative': 4,
        'ablative': 5,
        'prepositional': 6,
    }

    def __init__(self, target):
        self.target = target
        super(Inflection, self).__init__()

    def inflect(self, form_name):
        if form_name in self.forms:
            return self._inflect(self.forms[form_name])
        else:
            raise AttributeError('Unknown inflection form')

    def inflection(self, form_id):
        from warnings import warn
        warn('Deprecated method, use staff.inflections.inflect')
        return self._inflect(form_id)

    def _inflect(self, form_id):
        if isinstance(self.target, str):
            return self.inflect_text(form_id)
        else:
            return self.inflect_staff(form_id)

    def inflect_staff(self, form_id):
        from staff.person import metainflect as inflect

        return inflect.inflect_fio(
            first_name=self.target.i_first_name,
            last_name=self.target.i_last_name,
            form_id=form_id,
            gender=getattr(self.target, 'gender', GENDER.MALE)
        )

    def inflect_text(self, form_id):
        """
        Перевод фамилии и имени пользователя, переданных в виде входного текстового параметра.
        Этот способ перевода устарел. Перевод должен вызываться через staff объект.

        @todo: удалить метод, если его никто уже не использует
        """
        from staff.person import metainflect as inflect

        target = self.target
        is_brace = target.find('(') != -1
        is_minus = target.find('-') != -1

        if is_brace or is_minus:
            return self.inflect_double(target, form_id, '(' if is_brace else '-')
        else:
            return inflect.inflect(target, form_id, True) or target

    def inflect_double(self, target, form_id, delimiter='('):
        """Просклонять составное имя или фамилию.

        @param target: str строка
        @return: str
        """
        from staff.person import metainflect as inflect

        invert_delimiter = ')' if delimiter == '(' else ''
        before_delimiter, after_delimiter = [part.strip() for part in target.split(delimiter, 1)]
        if invert_delimiter:
            after_delimiter = after_delimiter.strip(invert_delimiter)
        after_delimiter = after_delimiter.strip()

        before_paren_inf = inflect.inflect(
            before_delimiter,
            form_id,
            True
        )
        if before_paren_inf:
            before_delimiter = before_paren_inf

        after_paren_inf = inflect.inflect(
            after_delimiter,
            form_id,
            True
        )
        if after_paren_inf:
            after_delimiter = after_paren_inf

        return ''.join((
            before_delimiter,
            ' ' if delimiter == '(' else '',
            delimiter,
            after_delimiter,
            invert_delimiter
        ))

    def __iter__(self):
        yield self.subjective
        yield self.genitive
        yield self.dative
        yield self.accusative
        yield self.ablative
        yield self.prepositional

    def __str__(self):
        return ', '.join([x for x in self])

    def __getattr__(self, attr_name):
        return self.inflect(attr_name)


class StaffInflection(Inflection):

    def inflection(self, form_id):
        return super(StaffInflection, self).inflection(form_id)

    def __init__(self, staff):
        self.staff = staff
        super(StaffInflection, self).__init__(staff)

    def __len__(self):
        return 1

    # NOTE: need to remove this method when all guys around the world
    # will use staff.inflections AS OBJECT, not as one-element list
    def __getitem__(self, key):
        if isinstance(key, int):
            return self
        else:
            return super(StaffInflection, self).__getitem__(key)
