import logging
from datetime import date, timedelta

from django.conf import settings
from django.template import Template

from staff.django_intranet_notifications import Notification
from staff.departments.models import Department
from staff.lib.db import atomic

from staff.person.models import Staff, AFFILIATION, WelcomeEmailTemplate


logger = logging.getLogger(name='person.notification')
YANDEX_MARKET_DEPARTMENT_URL = 'yandex_monetize_market'


def update_survey_mail_date(staff):
    HR_SURVEY_DELAY = settings.HR_SURVEY_DELAY
    today = date.today()

    from .controllers import Person
    person = Person(staff)
    person.date_survey_letter = today + timedelta(days=HR_SURVEY_DELAY)
    person.save()


@atomic
def _send_welcome(target, template):
    WelcomeMailNotification(
        person=target,
        template=template,
        target='WELCOME_MAILING',
        context={'person': target}
    ).send(
        recipients=[target.work_email],
    )
    target.extra.is_welcome_mail_sent = True
    target.extra.save()


def _get_welcome_mail_recipients():
    market_tree_id, market_lft, market_rght = (
        Department.objects
        .filter(intranet_status=1, url=YANDEX_MARKET_DEPARTMENT_URL)
        .values_list('tree_id', 'lft', 'rght')
        .get()
    )
    return (
        Staff.objects
        .select_related('extra')
        .filter(
            extra__is_welcome_mail_sent=False,
            is_dismissed=False,
            is_robot=False,
            affiliation=AFFILIATION.YANDEX,
        ).exclude(
            department__tree_id=market_tree_id,
            department__lft__gte=market_lft,
            department__rght__lte=market_rght,
        )
    )


def send_welcome_mail(person_ids=None):
    template = WelcomeEmailTemplate.objects.first()
    template = {
        'ru': template.text.replace('\r\n', '\n'),
        'en': template.text_en.replace('\r\n', '\n')
    }

    target_persons = _get_welcome_mail_recipients()

    if person_ids is not None:
        target_persons = target_persons.filter(id__in=person_ids)

    if not target_persons:
        return (
            'Welcome emails have already been sent to persons {} '
            'Aborting'.format(person_ids)
        )

    for person in target_persons:
        try:
            _send_welcome(
                target=person,
                template=template['ru' if person.lang_ui == 'ru' else 'en'],
            )
        except Exception:
            logger.exception(
                'Error trying to send welcome email to %s', person.login)


class SurveyMailNotification(Notification):
    template = 'mail_survey.html'

    def get_subj_id(self):
        if 'recipient' in self.context:
            return 'survey_email_for_%s' % self.context.get('recipient')
        return 'survey_email'


class WelcomeMailNotification(Notification):
    def __init__(self, person, template, *args, **kwargs):
        self.person = person
        self._template_text = template
        super(WelcomeMailNotification, self).__init__(*args, **kwargs)

    @property
    def template(self):
        return Template(self._template_text)

    def get_subj_id(self):
        return 'welcome_email_for_%s' % self.person.login


class PasswordReminderNotification(Notification):
    template = 'password_reminder.html'

    def get_subj_id(self):
        return 'password_reminder_%s' % date.today().isoformat()


def send_password_reminder_mails(recipients):
    msg = 'PASSWORD_REMINDER login "%s"'
    for email, context in recipients:
        try:
            PasswordReminderNotification(
                target='PASSWORD_REMINDER',
                context=context,
            ).send(
                recipients=[email],
            )
        except Exception:
            logger.exception(msg, email)
        else:
            logger.info(msg, email)
