import logging
from typing import Optional, Callable

from staff.lib import requests, blackbox

logger = logging.getLogger('staff.passport.base')


class PassportException(Exception):
    pass


class LoadException(PassportException):
    pass


class PushException(PassportException):
    pass


class ConvertException(PassportException):
    pass


class BBField(object):
    class Error(Exception):
        pass

    def __init__(
        self,
        passport_name: str,
        passport_field: Optional[str] = None,
        to_bb: Optional[Callable] = None,
        from_bb: Optional[Callable] = None,
    ):
        self.passport_name = passport_name
        self.dbfield = passport_field
        self.to_bb = to_bb or (lambda v: v)
        self.from_bb = from_bb or (lambda v: v)
        self.attribute = blackbox.ATTRIBUTES.get(passport_name)

        if passport_field is None and self.attribute is None:
            logger.error('Attribute not found for passport name %s', passport_name)
            raise self.Error('Attribute not found for passport name %s', passport_name)

    def __get__(self, instance, owner):
        value = instance.bb_data.get(self.passport_name)
        return self.from_bb(value)

    def __set__(self, instance, value):
        if self.__get__(instance, None) == value:
            return
        try:
            value = self.to_bb(value)
        except TypeError:
            value = self.to_bb(value, instance)
        instance.for_update[self.passport_name] = value

    def set_name(self, name):
        self.name = name


class PassportMeta(type):
    def __new__(mcs, name, bases, attrs):
        passport_fields = {}
        passport_attributes = {}
        for attr_name, value in attrs.items():
            if isinstance(value, BBField):
                value.set_name(attr_name)
                if value.dbfield is not None:
                    passport_fields[value.dbfield] = value.passport_name
                if value.attribute is not None:
                    passport_attributes[value.attribute] = value.passport_name

        attrs['passport_fields'] = passport_fields
        attrs['passport_attributes'] = passport_attributes
        return super(PassportMeta, mcs).__new__(mcs, name, bases, attrs)


class Passport(metaclass=PassportMeta):

    CONSUMER_PARAM = {'consumer': 'yastaff'}

    def __init__(self, uid):
        self.uid = uid
        self._bb_data = None
        self.for_update = {}

    @property
    def bb_data(self):
        if self._bb_data is None:
            self._bb_data = self.extract_bb_data(self.load_bb_data())
        return self._bb_data

    def save(self):
        if self.for_update:
            self.upload_bb_data()

    def load_bb_data(self):
        try:
            answer = self.blackbox.get_user_info(
                uid=self.uid,
                passport_fields=self.passport_fields,
                attributes=self.passport_attributes,
            )
        except blackbox.BlackboxError:
            logger.exception(
                'Exception when load data for uid %s from %s',
                self.uid,
                self.__class__.__name__,
            )
            raise LoadException

        if not answer['uid']:
            logger.error(
                'Uid "%s" not found in %s',
                self.uid,
                self.__class__.__name__,
            )
            raise LoadException

        return answer

    def extract_bb_data(self, answer):
        try:
            return dict(
                **{
                    self.passport_fields[k]: v
                    for k, v in answer.get('dbfields', {}).items()
                    if k in self.passport_fields
                },
                **{
                    self.passport_attributes[int(k)]: v
                    for k, v in answer.get('attributes', {}).items()
                    if int(k) in self.passport_attributes
                },
            )
        except KeyError or blackbox.BlackboxError:
            logger.exception(
                'Unexpected answer for uid %s from %s: %s',
                self.uid,
                self.__class__.__name__,
                answer,
            )
            raise ConvertException

    def _passport_api_request(self, method, url, data):
        try:
            answer = requests.request(
                method=method,
                url=url,
                params=self.CONSUMER_PARAM,
                timeout=self.TIMEOUT,
                data=data,
            )
        except requests.Timeout:
            logger.error(
                'Timeout on connect to the %s',
                self.__class__.__name__,
            )
            raise PassportException
        except Exception:
            logger.exception(
                'Unexpected exception on request to the %s with data: %s',
                self.__class__.__name__,
                data,
            )
            raise PassportException

        error = 'No Grants' in answer.text or answer.status_code != 200

        log = logger.error if error else logger.info

        log(
            'passport request data: %s; %s answered: code: %s, content: %s',
            data,
            self.__class__.__name__,
            answer.status_code,
            answer.text,
        )

        if error:
            raise PassportException(answer.text)

    def push_to_passport(self, params, **kwargs):
        method = 'delete' if params is None else 'post'

        try:
            self._passport_api_request(method, self.push_passport_path(**kwargs), params)
        except PassportException as e:
            raise PushException(str(e))

    def push_passport_path(self, **kwargs):
        return self.PASSPORT_URL + self.PASSPORT_PATH.format(uid=self.uid, **kwargs)

    def upload_bb_data(self):
        raise NotImplementedError
