from staff.reports.objects.base import Report
from staff.reports.sources import get_departments, get_oebs_full_names
from staff.reports.utils import populate_departments

from staff.person.models import Staff, EMPLOYMENT


class StaffReport(Report):
    sheet_name = 'employees'

    person_fields = [
        'id',
        'first_name',
        'first_name_en',
        'last_name',
        'last_name_en',
        'middle_name',
        'login',
        'position',
        'position_en',
        'employment',
        'is_dismissed',
        'department_id',
        'department__name',
        'department__name_en',
        'office__name',
        'organization__name',
        'join_at',
        'quit_at',
        'gender',
        'birthday',
    ]

    person_filters = {
        'is_robot': False,
    }

    def get_head_rows(self):
        yield [
            'Имя',
            'Фамилия',
            'Отчество',
            'ФИО (по данным ОЕБС)',
            'ФИО (англ.)',
            'Логин',
            'Должность',
            'Должность (англ.)',
            'Офис',
            'Организация',
            'Форма занятости',
            'Уволен',
            'Подразделение',
            'Подразделение (англ.)',
            'Цепочка подразделений',
            'Цепочка подразделений (англ.)',
            'Дата приёма на работу',
            'Дата увольнения',
            'Пол',
            'Дата рождения',
        ]

    def join_data(self, oebs_full_names=None, _populated_departments=None, persons=None):
        if oebs_full_names is None:
            oebs_full_names = get_oebs_full_names()

        if _populated_departments is None:
            dep_qs = get_departments()
            _populated_departments = populate_departments(dep_qs)

        departments_paths = {
            dep_id: ' => '.join([d['name'] for d in dep['bread_crumb']])
            for dep_id, dep in _populated_departments.items()
        }
        departments_paths_en = {
            dep_id: ' => '.join([d['name_en'] for d in dep['bread_crumb']])
            for dep_id, dep in _populated_departments.items()
        }

        if persons is None:
            persons = (
                Staff.objects.values(*self.person_fields)
                .filter(**self.person_filters)
                .order_by('join_at')
            )

        for person_data in persons:
            person_data['oebs_full_name'] = oebs_full_names.get(person_data.pop('id'), '')

            dep_id = person_data.pop('department_id', '')
            person_data['department_path'] = departments_paths.get(dep_id, str(dep_id))
            person_data['department_path_en'] = departments_paths_en.get(dep_id, str(dep_id))

            person_data['employment'] = str(EMPLOYMENT._text_values.get(person_data['employment'], ''))

            yield person_data

    def get_rows(self):
        persons_data = self.join_data()

        for person_data in persons_data:
            yield [
                person_data['first_name'],
                person_data['last_name'],
                person_data['middle_name'],
                person_data['oebs_full_name'],
                '{first} {last}'.format(
                    first=person_data['first_name_en'],
                    last=person_data['last_name_en'],
                ),
                person_data['login'],
                person_data['position'],
                person_data['position_en'],
                person_data['office__name'],
                person_data['organization__name'],
                person_data['employment'],
                self.yes_no_option(person_data['is_dismissed']),
                person_data['department__name'],
                person_data['department__name_en'],
                person_data['department_path'],
                person_data['department_path_en'],
                self.str_date(person_data['join_at']),
                self.str_date(person_data['quit_at']),
                self.gender(person_data['gender']),
                self.str_date(person_data['birthday']),
            ]

    @staticmethod
    def str_date(date):
        return date.strftime('%d.%m.%Y') if date else ''

    @staticmethod
    def yes_no_option(val):
        return 'Да' if val else 'Нет'

    @staticmethod
    def gender(g):
        return {'M': 'М', 'F': 'Ж'}.get(g, 'М')
