from datetime import datetime

from django.db.models import Q

from staff.reports.sources import get_value_streams
from staff.reports.utils import populate_departments

from staff.person.models import AFFILIATION, Staff
from staff.person.reports.objects.staff_report import StaffReport


class StaffUmbrellasReport(StaffReport):
    person_fields = [
        'id',
        'first_name',
        'first_name_en',
        'last_name',
        'last_name_en',
        'middle_name',
        'login',
        'position',
        'position_en',
        'employment',
        'is_dismissed',
        'department_id',
        'department__name',
        'department__name_en',
        'office__name',
        'organization__name',
        'join_at',
        'quit_at',
        'gender',
        'birthday',
        'assignments__value_stream_id',
        'assignments__value_stream__name',
        'assignments__value_stream__name_en',
        'umbrellas__umbrella__goal_id',
        'umbrellas__umbrella__name',
        'umbrellas__engagement',
    ]

    person_filters = {
        'is_robot': False,
        'is_dismissed': False,
        'affiliation': AFFILIATION.YANDEX,
        'assignments__main_assignment': True,
    }

    def get_head_rows(self):
        yield [
            'Имя',
            'Фамилия',
            'Отчество',
            'ФИО (по данным ОЕБС)',
            'ФИО (англ.)',
            'Логин',
            'VS',
            'Цепочка VS',
            'VS (англ.)',
            'Цепочка VS (англ.)',
            'Зонт',
            'Ссылка на зонт',
            '% распределения на зонт',
            'Должность',
            'Должность (англ.)',
            'Офис',
            'Организация',
            'Форма занятости',
            'Уволен',
            'Подразделение',
            'Подразделение (англ.)',
            'Цепочка подразделений',
            'Цепочка подразделений (англ.)',
            'Дата приёма на работу',
            'Дата увольнения',
            'Пол',
            'Дата рождения',
        ]

    def get_rows(self):
        persons = (
            Staff.objects.values(*self.person_fields)
            .filter(**self.person_filters)
            .filter(Q(umbrellas__engaged_to__isnull=True) | Q(umbrellas__engaged_to__gte=datetime.now()))
            .order_by('join_at')
        )

        populated_value_streams = populate_departments(get_value_streams())
        value_streams_paths = {
            dep_id: ' => '.join([d['name'] for d in dep['bread_crumb']])
            for dep_id, dep in populated_value_streams.items()
        }
        value_streams_paths_en = {
            dep_id: ' => '.join([d['name_en'] for d in dep['bread_crumb']])
            for dep_id, dep in populated_value_streams.items()
        }

        persons_data = self.join_data(persons=persons)

        for person_data in persons_data:
            value_stream_id = person_data.get('assignments__value_stream_id', '')
            goal_id = person_data['umbrellas__umbrella__goal_id']
            engagement = person_data['umbrellas__engagement']
            umbrella_name = person_data['umbrellas__umbrella__name']
            goal_url = ('https://goals.yandex-team.ru/compilations/?goal=' + str(goal_id)) if goal_id else ''

            if goal_id is None and engagement is not None:
                umbrella_name = '(Работает на VS)'

            yield [
                person_data['first_name'],
                person_data['last_name'],
                person_data['middle_name'],
                person_data['oebs_full_name'],
                '{first} {last}'.format(first=person_data['first_name_en'], last=person_data['last_name_en']),
                person_data['login'],
                person_data['assignments__value_stream__name'],
                value_streams_paths.get(value_stream_id, str(value_stream_id)),
                person_data['assignments__value_stream__name_en'],
                value_streams_paths_en.get(value_stream_id, str(value_stream_id)),
                umbrella_name or '',
                goal_url,
                engagement or '',
                person_data['position'],
                person_data['position_en'],
                person_data['office__name'],
                person_data['organization__name'],
                person_data['employment'],
                self.yes_no_option(person_data['is_dismissed']),
                person_data['department__name'],
                person_data['department__name_en'],
                person_data['department_path'],
                person_data['department_path_en'],
                self.str_date(person_data['join_at']),
                self.str_date(person_data['quit_at']),
                self.gender(person_data['gender']),
                self.str_date(person_data['birthday']),
            ]
