import logging
from datetime import date
from hashlib import sha1
from django.conf import settings

from staff.person.models import Staff

from staff.person import notifications

logger = logging.getLogger('emails.survey_email')
SURVEY_URL = settings.SURVEY_URL


def can_send_letter(params):
    return any([
        params.get('helpdesk'),
        params.get('cadre'),
        params.get('adaptation'),
    ])


def get_survey_mail_recipients():
    return (
        Staff.objects
        .select_related('office__settings',)
        .filter(extra__date_survey_letter=date.today())
    )


def get_office_services(person):
    try:
        office_settings = person.office.settings
    except AttributeError:
        return {}
    if not office_settings:
        return {}
    return {
        'helpdesk': office_settings.helpdesk_service,
        'cadre': office_settings.cadre_service,
        'adaptation': office_settings.adaptation_service
    }


def send_mails():
    from staff.person.controllers import Person

    for staff in get_survey_mail_recipients():
        params = get_office_services(staff)
        try:
            if can_send_letter(params):
                params['hash'] = sha1(staff.work_email.encode('utf-8')).hexdigest()

                context = {
                    'recipient': staff.first_name,
                    'survey_url': SURVEY_URL.format(**params),
                }
                notifications.SurveyMailNotification(
                    target='SURVEY_MAILING',
                    context=context,
                ).send(
                    recipients=[staff.work_email],
                )

            person = Person(staff)
            person.date_survey_letter = None
            person.save()
        except Exception:
            logger.exception('Failed to send survey email for %s', staff.login)
