import itertools
import logging
from collections import defaultdict
from datetime import datetime, timedelta, date

from staff.celery_app import app
from staff.gap.controllers.gap import GapQueryBuilder, GapCtl
from staff.gap.controllers.utils import replace_useless_hmst
from staff.gap.workflows.choices import GAP_STATES
from staff.lib.tasks import LockedTask
from staff.map.models import COUNTRY_CODES
from staff.person.models import Staff
from staff.person.tasks import FULL_YANDEX

logger = logging.getLogger(__name__)

acceptable_states = [GAP_STATES.NEW, GAP_STATES.SIGNED, GAP_STATES.CONFIRMED]
fields = ['id', 'person_id', 'date_from', 'date_to', 'state']


class GapFilter(object):
    def __init__(self, person_set, from_date, to_date):
        self.person_set = person_set
        self.from_date = from_date
        self.to_date = to_date


def get_maternity_days(gap_filter, last_accrual_dates):
    """
    Посчитать количество дней отпусков по уходу за ребенком.
    Для сотрудников в person_set, from_date <= день отпуска <= to_date
    """
    gqb_maternity = (
        GapQueryBuilder()
        .workflow('maternity')
        .person_ids([person.id for person in gap_filter.person_set])
        .date_interval_intersect(gap_filter.from_date, gap_filter.to_date)
        .op_in('state', acceptable_states)
    )
    maternity_gaps = GapCtl().find_gaps(query=gqb_maternity.query(), fields=fields)

    not_counted = defaultdict(int)
    for gap in maternity_gaps:
        person_id = gap['person_id']
        date_from = max(gap['date_from'], last_accrual_dates[person_id])
        date_to = min(gap['date_to'], gap_filter.to_date)

        if date_from <= date_to:
            not_counted[person_id] += (date_to - date_from).days + 1

    return not_counted


def get_absence_days(gap_filter, last_accrual_dates):
    """
    Посчитать количество дней прогулов.
    Для сотрудников в person_set, from_date <= день прогула <= to_date
    """
    gqb_absence = (
        GapQueryBuilder()
        .workflow('absence')
        .work_in_absence(False)
        .person_ids([person.id for person in gap_filter.person_set])
        .date_interval_intersect(gap_filter.from_date, gap_filter.to_date)
        .op_in('state', acceptable_states)
    )

    absence_gaps = GapCtl().find_gaps(query=gqb_absence.query(), fields=fields + ['full_day'])

    not_counted = defaultdict(int)
    for gap in absence_gaps:
        person_id = gap['person_id']
        if gap['full_day']:
            date_from = max(gap['date_from'], last_accrual_dates[person_id])
            date_to = min(gap['date_to'], gap_filter.to_date)
        else:
            date_from = max(
                last_accrual_dates[person_id],
                replace_useless_hmst(gap['date_from']) + timedelta(days=1)
            )
            date_to = min(gap_filter.to_date, replace_useless_hmst(gap['date_to']) - timedelta(days=1))

        if date_from <= date_to:
            not_counted[person_id] += (date_to - date_from).days + 1

    return not_counted


def get_excluded_days(gap_filter, data):
    maternity_days = get_maternity_days(gap_filter, data)
    absence_days = get_absence_days(gap_filter, data)

    # { person_id: number_of_days }
    not_counted = defaultdict(int)
    for key, val in itertools.chain(maternity_days.items(), absence_days.items()):
        not_counted[key] += val

    return not_counted


def update_vacations(exec_date):
    today = exec_date

    possible_accrual_date = today - timedelta(days=15)
    persons = (
        Staff.objects
        .filter(affiliation__in=FULL_YANDEX)
        .filter(office__city__country__code=COUNTRY_CODES.BELARUS)
        .filter(extra__last_vacation_accrual_at__lte=possible_accrual_date)
    )

    if persons.count() == 0:
        return

    data = persons.values_list('id', 'extra__last_vacation_accrual_at')
    last_accrual_dates = {person_id: datetime(d.year, d.month, d.day) for person_id, d in data}

    from_date = min(last_accrual_dates.values())
    to_date = datetime(today.year, today.month, today.day) - timedelta(days=1)

    gap_filter = GapFilter(
        person_set=persons,
        from_date=from_date,
        to_date=to_date
    )

    not_counted = get_excluded_days(gap_filter, last_accrual_dates)

    # среднемесячное количество дней
    avg_day_count = 29.7
    # среднемесячное начисление отпусков
    # 25 календарных дней / 12 месяцев = 2,08(3) дня
    accrual_per_month = 2.083

    for person in persons:
        _update_belarus_person_vacation(accrual_per_month, avg_day_count, not_counted, person, today)


def _update_belarus_person_vacation(accrual_per_month, avg_day_count, not_counted, person, today):
    try:
        logger.info("Processing vacation accrual for %s", person.login)
        last_vacation_accrual = person.extra.last_vacation_accrual_at or person.join_at
        days_passed = (today - last_vacation_accrual).days
        accrual = (days_passed - not_counted[person.id]) / avg_day_count * accrual_per_month

        current_vacation_days = person.vacation or 0.0
        person.vacation = current_vacation_days + accrual
        person.extra.last_vacation_accrual_at = today
        person.extra.save()
        person.save()
        logger.info("Processed vacation accrual for %s +%.3f =%.3f days", person.login, accrual, person.vacation)
    except Exception as e:
        logger.exception("Error processing vacation accrual for %s: %s", person.login, e)
        raise


@app.task(ignore_result=True)
class UpdateVacationForBelarus(LockedTask):
    """
    Таска считает сколько за последние 15 дней
    было отработанных дней (исключаются из них: отпуски по уходу за ребенком и дни прогулы)
    и начисляет отпускные дни за этот период
    """
    def locked_run(self, *args, **kwargs):
        update_vacations(exec_date=date.today())
