import time
import logging
from hashlib import md5

import yenv
from django.conf import settings

from staff.person_avatar.tasks import CheckUserPhotos
from staff.person_avatar.models import AvatarMetadata
from staff.person_avatar.storage import AvatarStorage, AvatarError

MAIN, AVATAR, GRAVATAR = 'main', 'avatar', 'gravatar'

logger = logging.getLogger(__name__)


class AvatarCollectionError(Exception):
    pass


class BaseAvatarCollection(object):
    avatar_class = None

    def __init__(self, owner=None):
        self.owner = owner
        self.owner_avatars = None
        self._init_owner_avatars()

    def __iter__(self):
        for avatar in self.owner_avatars:
            yield avatar

    def __getitem__(self, key):
        return self.owner_avatars[key]

    def _init_owner_avatars(self):
        qs = AvatarMetadata.objects.filter(
            **{
                self.avatar_class.metadata_owner_field: self.owner,
                'is_deleted': False,
            }
        )
        self.owner_avatars = [
            self._wrap(avatar_metadata)
            for avatar_metadata in qs.order_by('id')
        ]

    def get_by_field(self, field, value):
        for avatar in self.owner_avatars:
            if getattr(avatar, field) == value:
                return avatar
        raise AvatarCollectionError(
            'No {class_name} with {field}={val} for owner {owner}'.format(
                class_name=self.avatar_class.__name__,
                field=field,
                val=value,
                owner=self.owner.id,
            )
        )

    def get(self, pk):
        return self.get_by_field('id', pk)

    def _wrap(self, db_instance):
        return self.avatar_class(
            owner=getattr(db_instance, self.avatar_class.metadata_owner_field),
            avatar_metadata=db_instance,
        )

    def first(self, exclude_pk=None):
        if not self.exists():
            return None

        if exclude_pk is None:
            return self[0]

        first_avatar = [
            av for av in self.owner_avatars[:2]
            if av.id != exclude_pk
        ]
        first_avatar = first_avatar[0] if first_avatar else None

        return first_avatar

    def count(self):
        return len(self.owner_avatars)

    def exists(self):
        return self.count() > 0

    def upload(self, picture_url=None, picture_file=None):
        new_avatar = self.avatar_class(owner=self.owner)
        new_avatar.upload(picture_url, picture_file)
        self._init_owner_avatars()
        self.post_upload(new_avatar)

    def post_upload(self, new_avatar):
        pass

    def delete(self, pk):
        avatar = self.get(pk)
        first_avatar = self.first(exclude_pk=pk)

        if first_avatar:
            first_avatar_attrs = set()
            if avatar.is_main:
                first_avatar_attrs.add(MAIN)
                storage_id = avatar.avatar_id(attr='main')
                avatar.delete_from_storage(storage_id)
            if avatar.is_avatar:
                first_avatar_attrs.add('avatar')
                storage_id = avatar.avatar_id(attr=AVATAR)
                avatar.delete_from_storage(storage_id)

            self.make(first_avatar.id, first_avatar_attrs)
        avatar.mark_deleted()
        avatar.delete_from_storage()
        self._init_owner_avatars()

    def make(self, pk, marks):
        marks = {'main', 'avatar'} & set(marks)
        avatar = self.get(pk)
        for mark in marks:
            try:
                metadata_attr = 'is_' + mark
                marked_avatar = self.get_by_field(metadata_attr, True)
                marked_avatar.unmake(mark)
                # marked_avatar.delete_from_storage(avatar.avatar_id(attr=mark))
            except AvatarCollectionError:
                pass
        avatar.mark_and_make(marks)
        self._init_owner_avatars()


class BaseAvatar(object):

    UPLOAD_RETRIES = settings.AVATAR_UPLOAD_RETRIES
    SLEEP_SEC = settings.AVATAR_UPLOAD_SLEEP_SEC
    AVATAR_SCOPE = 'staff'

    metadata_owner_field = ''

    def __init__(self, owner=None, avatar_metadata=None):
        self.owner = owner
        self.storage = AvatarStorage(scope=self.AVATAR_SCOPE)
        if avatar_metadata:
            self.metadata = avatar_metadata
        else:
            self.metadata = AvatarMetadata(
                **{self.metadata_owner_field: self.owner}
            )

    def __getattr__(self, attr):
        return getattr(self.metadata, attr)

    @property
    def owner_work_email(self):
        """Should return owner's work email"""
        raise NotImplementedError

    @property
    def owner_login(self):
        """Should return owner's login"""
        raise NotImplementedError

    def get_extrafields_instance(self):
        raise NotImplementedError

    @staticmethod
    def gravatar_id(email):
        return md5(email.encode('utf-8')).hexdigest()

    def avatar_id(self, attr=None):
        if self.metadata.id is None:
            raise Exception('Metadata is unsaved')

        if attr == 'gravatar':
            if self.owner_work_email:
                return self.gravatar_id(self.owner_work_email)
            else:
                return None

        if attr is not None:
            return '{login}-{attr}'.format(login=self.owner_login, attr=attr)
        else:
            return str(self.metadata.id)

    def _execute(self, method, *args, **kwargs):
        error = None
        for i in range(self.UPLOAD_RETRIES):
            try:
                method(*args, **kwargs)
                return
            except AvatarError as e:
                error = e
                time.sleep(self.SLEEP_SEC)
        raise error

    def upload(self, picture_url=None, picture_file=None):
        try:
            logger.debug(
                'Create metadata %s', self.owner_login
            )
            self.metadata.is_deleted = False
            self.metadata.save()
        except Exception:
            logger.exception('Failed create metadata %s', self.owner_login)
            raise

        avatar_id = self.avatar_id()

        try:
            if picture_url is not None:
                logger.info('Upload %s url %s', avatar_id, picture_url)
                self._execute(
                    self.storage.upload_by_url,
                    avatar_id,
                    picture_url,
                )
            if picture_file is not None:
                logger.info('Upload %s file', avatar_id)
                self._execute(
                    self.storage.upload_by_file,
                    avatar_id,
                    picture_file,
                )
        except AvatarError:
            logger.info('Failed upload new avatar %s', avatar_id)
            self.metadata.is_deleted = True
            self.metadata.save()
            raise
        except Exception:
            logger.exception('Failed upload new avatar %s', avatar_id)
            self.metadata.is_deleted = True
            self.metadata.save()
            raise

    def mark_and_make(self, attrs):
        attrs = set(attrs)
        if AVATAR in attrs:
            attrs.add(GRAVATAR)
        for attr in attrs:
            self._mark(attr)
        if attrs:
            kwargs = {
                'owner_id': self.owner.id,
                'owner_type': self.metadata_owner_field,
            }

            if yenv.type == 'development':
                CheckUserPhotos(**kwargs)
            else:
                CheckUserPhotos.apply_async(
                    kwargs={
                        'owner_id': self.owner.id,
                        'owner_type': self.metadata_owner_field,
                    },
                    countdown=10
                )

    def _mark(self, attr):
        attr_img = attr + '_img_for_upload'
        metadata_attr = 'is_' + attr if attr != 'gravatar' else 'is_avatar'

        extra = self.get_extrafields_instance()

        setattr(extra, attr_img, self.metadata)
        extra.save()

        setattr(self.metadata, metadata_attr, True)
        self.metadata.save()

    def make(self, attr):
        """to call from UpdatePhotos task only"""
        self._execute(
            self.storage.upload_by_url,
            self.avatar_id(attr=attr),
            self.url,
        )

    def unmake(self, attr):
        metadata_attr = 'is_' + attr if attr != 'gravatar' else 'is_avatar'

        base_avatar_id = self.avatar_id()
        setattr(self.metadata, metadata_attr, False)
        logger.debug(
            'Update metadata, unset "%s" from avatar %s',
            attr,
            base_avatar_id,
        )
        self.metadata.save()

    def mark_deleted(self):
        self.metadata.is_deleted = True
        self.metadata.save()

    def delete_from_storage(self, avatar_id=None):
        avatar_id = str(avatar_id or self.id)
        try:
            logger.debug('Delete avatar %s', avatar_id)
            self._execute(self.storage.delete, avatar_id)
        except Exception:
            logger.exception('Failed deleting avatar %s from storage', avatar_id)

    @property
    def url(self):
        return '{avatar_url}{avatar_id}/orig'.format(
            avatar_url=settings.AVATAR_URL.format(scope=self.AVATAR_SCOPE),
            avatar_id=self.avatar_id()
        )
