import asyncio
from hashlib import md5
from itertools import chain

import aiohttp
from django.conf import settings
from django.core.management.base import BaseCommand

from staff.lib.tvm2 import async_get_tvm_ticket
from staff.rfid.models import Badge
from staff.person.models import Staff
from staff.preprofile.models import Preprofile
from staff.person_avatar.models import AvatarMetadata

RPS = 100

session = aiohttp.ClientSession()

missed_avatars = []
to_upload_avatars = []


class Command(BaseCommand):
    help = 'Мигрирует аватарки из старой в новую аватарницу'

    def handle(self, *args, **kwargs):
        migrate()


def migrate():
    loop = asyncio.get_event_loop()

    loop.run_until_complete(migrate_meta_avatars())
    loop.run_until_complete(migrate_staff_avatars())
    loop.run_until_complete(migrate_badge_avatars())
    loop.run_until_complete(migrate_gravatars())

    loop.run_until_complete(session.close())


def migrate_meta_avatars():
    avatar_ids = AvatarMetadata.objects.values_list('id', flat=True)
    return migrate_by_avatar_ids('staff', avatar_ids)


def migrate_staff_avatars():
    staff_logins = Staff.objects.values_list('login', flat=True)
    staff_ids = list(chain([f'{login}-main' for login in staff_logins], [f'{login}-avatar' for login in staff_logins]))
    return migrate_by_avatar_ids('staff', staff_ids)


def migrate_preprofile_avatars():
    logins = Preprofile.objects.filter(login__isnull=False).order_by('-id').values_list('login', flat=True)
    ids = list(chain([f'{login}-main' for login in logins], [f'{login}-avatar' for login in logins]))
    return migrate_by_avatar_ids('staff', ids)


def migrate_badge_avatars():
    rfid_ids = Badge.objects.values_list('id', flat=True)
    return migrate_by_avatar_ids('rfid', rfid_ids)


def migrate_gravatars():
    emails = list(Staff.objects.filter(work_email__isnull=False).values_list('work_email', flat=True))
    preprofile_emails = [
        f'{login}@{domain}'
        for login, domain in (
            Preprofile.objects
            .filter(login__isnull=False, email_domain__isnull=False)
            .values_list('login', 'email_domain')
        )
    ]
    emails.extend(preprofile_emails)
    emails = [md5(email.encode('utf-8')).hexdigest() for email in emails]
    return migrate_by_avatar_ids('staff', emails)


async def migrate_by_avatar_ids(scope, avatar_ids):
    print(f'{len(avatar_ids)} avatars to migrate')
    tasks_by_avatar_id = {}
    for avatar_id in avatar_ids:
        tasks_by_avatar_id[avatar_id] = asyncio.ensure_future(migrate_image(scope, avatar_id))
        await asyncio.sleep(1/RPS)
    for index, avatar_id in enumerate(avatar_ids):
        done = await tasks_by_avatar_id[avatar_id]
        if index % 1000 == 0:
            print(f'{index} done', flush=True)
        if not done:
            missed_avatars.append((scope, str(avatar_id)))


async def migrate_image(scope: str, avatar_id: str) -> bool:
    image_url = f'http://center.yandex-team.ru/_debug/avatars-proxy/get-{scope}/{avatar_id}/orig'
    if scope != 'staff':
        avatar_id = f'{scope}-{avatar_id}'
    in_new = await in_new_avatar(avatar_id)
    if not in_new:
        to_upload_avatars.append(avatar_id)
        return await put_avatar(avatar_id, image_url)
    return in_new


async def put_avatar(avatar_id: str, image_url: str) -> bool:
    url = f'{settings.AVATAR_STORAGE_URL}/put-staff/{avatar_id}'
    token = await async_get_tvm_ticket('avatars-mds')
    async with session.get(url, headers={'X-Ya-Service-Ticket': token}, params={'url': image_url}) as response:
        if response.status not in [200, 434]:
            print(response.status, await response.text(), url)
            if response.status == 415:
                headers = {'X-Ya-Service-Ticket': token}
                params = {'url': image_url[:-4] + '300'}
                async with session.get(url, headers=headers, params=params) as response2:
                    if response2.status not in [200, 434]:
                        print(response2.status, await response2.text(), url)
                    return response2.status in [200, 434]
        return response.status in [200, 434]


async def in_new_avatar(avatar_id: str) -> bool:
    url = f'http://{settings.AVATAR_URL.format(scope="staff")}/{avatar_id}/orig'
    async with session.get(url) as response:
        return response.status != 404
