import socket
import logging

from django.conf import settings

from staff.lib import requests
from staff.lib import tvm2

logger = logging.getLogger('person_avatar.storage')


class AvatarError(Exception):
    pass


class AvatarConnectingError(AvatarError):
    pass


class AvatarUnexpectedError(AvatarError):
    pass


class AvatarBadRequestError(AvatarError):
    """Avatars responded with error"""
    def __init__(self, code):
        self.code = code


class AvatarStorageError(AvatarError):
    pass


class AvatarStorage:

    URL = settings.AVATAR_STORAGE_URL
    TIMEOUT = settings.AVATAR_UPLOAD_TIMEOUT

    def __init__(self, scope):
        self.scope = scope

    def _request_wrapper(self, action, *args, **kwargs):
        try:
            answer = action(
                timeout=self.TIMEOUT,
                headers={tvm2.TVM_SERVICE_TICKET_HEADER: tvm2.get_tvm_ticket_by_deploy('avatars-mds')},
                *args, **kwargs
            )

            self._log_result(answer)

            if 400 <= answer.status_code < 500 and answer.status_code != 404:
                raise AvatarBadRequestError(answer.status_code)
            if answer.status_code > 500:
                raise AvatarStorageError

            return answer

        except (requests.Timeout, socket.error):
            logger.exception(
                'Error connecting to AvatarStorage. Action:%s args:%s kwargs:%s',
                action.__name__,
                args,
                kwargs,
            )
            raise AvatarConnectingError
        except (AvatarBadRequestError, AvatarStorageError):
            logger.info(
                'Error communicating with AvatarStorage. Action:%s args:%s kwargs:%s',
                action.__name__,
                args,
                kwargs,
                exc_info=True,
            )
            raise
        except Exception as e:
            logger.exception(
                'An unexpected error while '
                'connecting to the Avatar.'
            )
            raise AvatarUnexpectedError(e)

    def _get(self, *args, **kwargs):
        return self._request_wrapper(requests.get, *args, **kwargs)

    def _post(self, *args, **kwargs):
        return self._request_wrapper(requests.post, *args, **kwargs)

    def _log_result(self, answer):
        msg_list = []
        msg_list.append('Avatar answer:')
        msg_list.append(' url = {0}'.format(answer.url))
        msg_list.append(' status code = {0}'.format(answer.status_code))
        if answer.text.strip():
            msg_list.append(' body = {0}'.format(answer.text.strip()))

        msg = '\n'.join(msg_list)
        logger.debug(msg)

    @property
    def storage_url(self):
        return self.URL

    def put_url(self, avatar_id):
        return f'{self.storage_url}/put-{self.scope}/{avatar_id}'

    def delete_url(self, avatar_id):
        return f'{self.storage_url}/delete-{self.scope}/{avatar_id}'

    def upload_by_file(self, avatar_id, picture_file):
        LATIN_FILENAME = 'user_photo_file'

        put_url = self.put_url(avatar_id)
        extension = ''
        if hasattr(picture_file, '_name'):
            extension = '.' + picture_file._name.split('.')[-1]
        picture_file._name = LATIN_FILENAME + extension
        files = {'file': picture_file}
        self._post(put_url, files=files)

    def upload_by_url(self, avatar_id, picture_url):
        put_url = self.put_url(avatar_id)
        params = {'url': picture_url}
        self._get(put_url, params=params)

    def delete(self, avatar_id):
        delete_url = self.delete_url(avatar_id)
        self._get(delete_url)
