from django.conf import settings
from django.core.exceptions import PermissionDenied
from django.utils.translation import ugettext as _
from django.views.decorators.csrf import csrf_exempt
from django.views.decorators.http import require_POST

from staff.lib.decorators import responding_json, available_for_external
from staff.person_avatar.controllers import MAIN, AVATAR, PersonAvatarCollection
from staff.person_profile.permissions.utils import get_block_registry


@require_POST
@csrf_exempt
@available_for_external
@responding_json
def make_main(request, avatar_id):
    avatar_id = int(avatar_id)
    person_avatars = _get_avatar_collection(request, avatar_id)
    person_avatars.make(avatar_id, marks=[MAIN])
    photos = person_avatars.owner_avatars
    return _get_avatar_response(person_avatars.owner, photos)


@require_POST
@csrf_exempt
@available_for_external
@responding_json
def make_avatar(request, avatar_id):
    avatar_id = int(avatar_id)
    person_avatars = _get_avatar_collection(request, avatar_id)
    person_avatars.make(avatar_id, marks=[AVATAR])
    photos = person_avatars.owner_avatars
    return _get_avatar_response(person_avatars.owner, photos)


@require_POST
@csrf_exempt
@available_for_external
@responding_json
def delete(request, avatar_id):
    avatar_id = int(avatar_id)
    person_avatars = _get_avatar_collection(request, avatar_id)
    person_avatars.delete(avatar_id)
    photos = person_avatars.owner_avatars
    return _get_avatar_response(person_avatars.owner, photos)


def _get_avatar_response(person, photos):
    main_exists = False
    for img in photos:
        if img.is_main:
            main = img
            main_exists = True
            break

    if main_exists:
        main = {
            'elem': 'img',
            'js': {
                'img': {
                    'id': main.pk,
                    'src': '//%s/api/v1/user/%s/photo_%d/300.jpg' % (
                        settings.CENTER_MASTER,
                        person.login,
                        main.pk,
                    ),
                },
            },
            'elemMods': {
                'size': 'big',
            },
            'src': '//%s/api/v1/user/%s/photo_%d/300.jpg' % (
                settings.CENTER_MASTER,
                person.login,
                main.pk,
            ),
        }
        controls = [
            {
                'block': 'b-form-layout',
                'elem': 'field',
                'elemMods': {
                    'right-margin': '4px',
                },
                'content': [
                    {
                        'block': 'b-form-checkbox',
                        'mods': {
                            'size': 'm',
                            'type': 'is-main',
                        },
                        'checkboxAttrs': {
                            'id': 'base',
                        },
                    },
                ],
            },
            {
                'block': 'b-form-layout',
                'elem': 'field',
                'elemMods': {
                    'right-margin': '8px',
                },
                'content': {
                    'block': 'b-form-element',
                    'content': {
                        'elem': 'label',
                        'attrs': {
                            'for': 'base',
                        },
                        'content': _("staff.anketa-edit.form.photo.set_main"),
                    },
                },
            },
            {
                'block': 'b-form-layout',
                'elem': 'field',
                'elemMods': {
                    'right-margin': '4px',
                },
                'content': [
                    {
                        'block': 'b-form-checkbox',
                        'mods': {
                            'size': 'm',
                            'type': 'is-avatar',
                        },
                        'checkboxAttrs': {
                            'id': 'ava',
                        },
                    },
                ],
            },
            {
                'block': 'b-form-layout',
                'elem': 'field',
                'elemMods': {
                    'right-margin': '16px',
                },
                'content': {
                    'block': 'b-form-element',
                    'content': {
                        'elem': 'label',
                        'attrs': {
                            'for': 'ava',
                        },
                        'content': _("staff.anketa-edit.form.photo.set_avatar"),
                    },
                },
            },
            {
                'block': 'b-form-layout',
                'elem': 'field',
                'content': {
                    'block': 'b-link',
                    'mods': {
                        'pseudo': 'yes',
                        'size': 'small',
                        'type': 'delete-photo',
                    },
                    'content': _("staff.anketa-edit.form.photo.delete"),
                },
            },
        ]
    else:
        main = {
            'elem': 'img',
            'elemMods': {
                'size': 'big',
            },
            'src': '//%s/media/i/nophoto.png' % settings.CENTER_MASTER,
        }
        controls = []

    others = None
    if len(photos) > 1:
        thumbs = []
        for img in photos:
            bem = {
                'elem': 'img',
                'elemMods': {
                    'size': 'small',
                },
                'js': {
                    'img': {
                        'id': img.pk,
                        'src': '//%s/api/v1/user/%s/photo_%d/300.jpg' % (
                            settings.CENTER_MASTER,
                            person.login,
                            img.pk,
                        ),
                    },
                },
                'src': '//%s/api/v1/user/%s/photo_%d/50/square.jpg' % (
                    settings.CENTER_MASTER,
                    person.login,
                    img.pk,
                ),
            }

            if img.is_main:
                bem['elemMods'].update({
                    'selected': 'yes',
                    'is-main': 'yes',
                })
            if img.is_avatar:
                bem['elemMods'].update({
                    'is-avatar': 'yes',
                })
            thumbs.append(bem)

        others = {
            'elem': 'others',
            'content': thumbs,
        }

    return {
        'block': 'b-avatar',
        'js': True,
        'content': [
            others,
            main,
            {
                'block': 'b-form-layout',
                'elem': 'line',
                'mix': [
                    {
                        'block': 'b-avatar',
                        'elem': 'control',
                    },
                ],
                'content': controls,
            },
        ],
    }


def _get_avatar_collection(request, avatar_id: int) -> PersonAvatarCollection:
    person_avatars = PersonAvatarCollection(by_metadata_id=avatar_id)

    if person_avatars.owner is None:
        if request.user.is_superuser:
            return person_avatars
        raise PermissionDenied('Avatar has no owner')

    target_login = person_avatars.owner.login
    permissions_ctl = get_block_registry(
            observer=request.user.get_profile(),
            login_or_logins=target_login,
            service_readonly=request.service_is_readonly,
        )

    if not permissions_ctl.is_available('upload_photo', target_login):
        raise PermissionDenied("You don't own this avatar")

    return person_avatars
