import sform
from datetime import datetime

from django.db.models import Q

from staff.gap.models import Workflow
from staff.gap.controllers.staff_utils import GapsByPersons

from ._base import lib, FieldFilter, BaseEqual, BaseIn, BaseExistence
from . import _forms


SELFPAID = 6


@lib.register
class GapFilter(FieldFilter):
    """Текущее отсутствие, равно/в списке, селект"""


@GapFilter.register
class Existence(BaseExistence):
    form_cls = _forms.TrueBooleanForm

    def q(self):
        person_ids = (
            GapsByPersons(datetime.now())
            .only_keys()
            .get_all('person_id')
        )

        q = Q(id__in=person_ids)
        return q if self.cleaned_data['value'] else ~q


class EqualForm(sform.SForm):
    workflow = sform.ModelChoiceField(
        queryset=Workflow.objects.exclude(id=SELFPAID),
        label_extractor='name',
        state=sform.REQUIRED,
    )
    will_work_in_absence = sform.BooleanField(default=False)

    def clean_will_work_in_absence(self, value):
        worklflow = self.cleaned_data['workflow']
        # флаг will_work_in_absence не будет учитываться для trip & conference_trip workflow_service
        if worklflow.name in ('trip', 'conference_trip'):
            self.cleaned_data['will_work_in_absence'] = False
        return self.cleaned_data['will_work_in_absence']


@GapFilter.register
class Equal(BaseEqual):
    form_cls = EqualForm

    def q(self):
        workflow = self.cleaned_data['workflow'].name
        work_in_absence = self.cleaned_data['will_work_in_absence']

        gaps_filter = (
            GapsByPersons(datetime.now())
            .filter_by_workflows([workflow])
            .only_keys()
        )

        if workflow not in ('trip', 'conference_trip'):
            gaps_filter.filter_by_work_in_absence(work_in_absence)

        person_ids = gaps_filter.get_all('person_id')
        return Q(id__in=person_ids)


@GapFilter.register
class Exclude(Equal):
    def q(self):
        return ~ super(Exclude, self).q()


class InForm(sform.SForm):
    workflows = sform.ModelMultipleChoiceField(
        queryset=Workflow.objects.exclude(id=SELFPAID),
        label_extractor='name',
        state=sform.REQUIRED,
    )


# @GapFilter.register
class In(BaseIn):
    form_cls = InForm

    def q(self):
        wfs = [d.name for d in self.cleaned_data['workflows']]
        person_ids = (
            GapsByPersons(datetime.now())
            .filter_by_workflows(wfs)
            .only_keys()
            .get_all('person_id')
        )

        return Q(id__in=person_ids)
