from typing import Optional, AnyStr, Sequence

from django.db.models import Q, Count
from django.db.models.query import QuerySet, ValuesQuerySet
from django.conf import settings

from staff.departments.models import Department
from staff.person.models import Staff
from staff.person_filter.controller import FilterCtl
from staff.users.models import User


class FilterContext(object):
    user: Optional[User] = None
    permission: Optional[AnyStr] = None

    def __init__(self, **kwargs):
        self.observer_tz = kwargs.get('observer_tz')
        self.user = kwargs.get('user')
        self.filter_id = kwargs.get('filter_id')
        self.filter_data = kwargs.get('filter_data')
        self.permission = kwargs.get('permission')

        if self.filter_id:
            self.saved_filter = FilterCtl.get_by_id(self.filter_id)
        elif self.filter_data:
            self.saved_filter = FilterCtl.get_by_data(self.filter_data)
        else:
            self.saved_filter = None

    dep_fields = (
        'id',
        'parent_id',
        'url',
        'tree_id',
        'lft',
        'rght',
        'name',
        'name_en',
        'level',
        'description',
        'description_en',
        'wiki_page',
        'maillists',
        'clubs',
        'position',
        'is_expanded',
        'is_hidden',
        'section_group_id',
        'section_caption_id',
        'tags',
    )

    def get_base_dep_qs(self, fields: Sequence[AnyStr] = None) -> ValuesQuerySet:
        if fields is None:
            fields = self.dep_fields

        qs = (
            Department.objects
            .filter(intranet_status=1)
            .exclude(id=settings.INTRANET_DISMISSED_STAFF_DEPARTMENT_ID)
            .values(*fields)
        )
        if self.permission:
            qs = qs.filter(self._departments_by_outstaff_perm_and_their_ancestors())
        return qs

    def _departments_by_outstaff_perm_and_their_ancestors(self) -> Q:
        if self.user.get_profile().is_internal():
            return Q()

        return (
            self.user.get_profile().departments_by_perm_query(self.permission, True)
            | self.user.get_profile().departments_by_perm_query(self.permission, False)
        )

    short_person_fields = (
        'id',
        'login',
        'first_name',
        'first_name_en',
        'last_name',
        'last_name_en',
        'department_id',
    )

    person_fields = short_person_fields + (
        'work_phone',
        'work_email',
        'position',
        'position_en',
        'office__id',
        'office__name',
        'office__name_en',
        'office__filter_id',
        'organization__name',
        'birthday',
        'join_at',
        'stafflastoffice__office_name',
        'stafflastoffice__office_name_en',
        'stafflastoffice__is_vpn',
        'stafflastoffice__updated_at',
        'room__num',
        'room__floor__num',
        'table__num',
        'home_phone',
        'work_mode',
    )

    def get_person_obj_qs(self) -> QuerySet:
        qs = Staff.objects.filter(is_dismissed=False)

        if self.saved_filter:
            assert self.saved_filter.is_valid()
            qs = qs.filter(self.saved_filter.get_query())

        if self.permission:
            qs = qs.filter(
                self.user
                .get_profile()
                .departments_by_outstaff_perm_query(self.permission, 'department__')
            )

        return qs

    def get_person_qs(self, fields: Sequence[AnyStr] = None) -> ValuesQuerySet:
        if fields is None:
            fields = self.short_person_fields
        qs = self.get_person_obj_qs().values(*fields)
        return qs

    def get_base_person_qs(self) -> ValuesQuerySet:
        return self.get_person_qs(self.person_fields)

    def get_persons_qty_qs(self) -> ValuesQuerySet:
        qs = (
            self.get_person_qs()
            .values('department_id')
            .annotate(qty=Count('id'))
        )
        return qs
