import yenv
import hashlib
import logging
from datetime import datetime

from staff.person.models import Staff

from staff.lib.models.mptt import filter_by_heirarchy

from .controller import FilterCtl
from .models import PersonsFilter, SavedPersonsFilter, PersonToPersonFilter
from .tasks import UpdatePersonsFilters

logger = logging.getLogger(__name__)


class SavedFilterCtl(object):
    KEY_FIELDS = 'person', 'persons_filter'
    FIELDS = tuple(
        set(SavedPersonsFilter._meta.get_all_field_names()) - set(KEY_FIELDS)
    )
    SUBSCRIPTION_FIELDS = [f for f in FIELDS if f.endswith('_subscription')]
    SUBSCRIPTIONS = [f[:-13] for f in SUBSCRIPTION_FIELDS]

    VALUES_QUERYSET_FIELDS = (
        'id',
        'name',
        'is_bookmark',
        'is_for_cab',
        'absences_subscription',
        'birthdays_subscription',
        'persons_filter__filter_id',
        'persons_filter__department__url',
    )

    def __init__(self, person):
        self.person = person

    def get_qs(self, saver_filter_id=None):
        qs = SavedPersonsFilter.objects.filter(person=self.person)
        if saver_filter_id is not None:
            qs = qs.filter(id=saver_filter_id)
        return qs

    def _format_sf(self, sf):
        sf = dict(sf)
        dep_url = sf.pop('persons_filter__department__url')
        if dep_url:
            sf['department_url'] = dep_url
        filter_id = sf.pop('persons_filter__filter_id')
        if filter_id:
            sf['filter_id'] = filter_id
        return sf

    def _get(self, **kwargs):
        qs = (
            self.get_qs().filter(**kwargs)
            .values(*self.VALUES_QUERYSET_FIELDS)
        )
        try:
            return qs.get()
        except SavedPersonsFilter.DoesNotExist:
            return None

    def get(self, saved_filter_id):
        saved_filter = self._get(id=saved_filter_id)
        if saved_filter:
            return self._format_sf(saved_filter)

    def get_by_department_filter_id(self, department_url, filter_id):
        saved_filter = self._get(
            persons_filter__department__url=department_url,
            persons_filter__filter_id=filter_id,
        )
        if saved_filter:
            return self._format_sf(saved_filter)

    def get_list(
            self,
            contains_login=None,
            is_bookmark=None,
            is_for_cab=None,
            absences_subscription=None,
            birthdays_subscription=None,
    ):
        qs = self.get_qs()
        if contains_login:
            qs = qs.filter(persons_filter__persons__login=contains_login)

        if is_bookmark is not None:
            qs = qs.filter(is_bookmark=is_bookmark)
        if is_for_cab is not None:
            qs = qs.filter(is_for_cab=is_for_cab)
        if absences_subscription is not None:
            qs = qs.filter(absences_subscription=absences_subscription)
        if birthdays_subscription is not None:
            qs = qs.filter(birthdays_subscription=birthdays_subscription)

        result = qs.values(*self.VALUES_QUERYSET_FIELDS)

        return [self._format_sf(r) for r in result]

    def create(self, data, with_task=True):
        filter_id = data['filter_id']
        if data['filter_id'] is None and data['department'] is None:
            filters = [{
                'operator': 'and',
                'field': 'PersonFilter',
                'condition': 'Equal',
                'person': data['person'].id,
            }]

            filter_ctl = FilterCtl.get_by_data(data={'filters': filters})
            filter_id = filter_ctl.create_id()

        person_filter = PersonsFilter.objects.get_or_create(
            filter_id=filter_id,
            department=data['department'],
            person=data['person'],
        )[0]

        obj = SavedPersonsFilter.objects.get_or_create(
            person=self.person,
            persons_filter=person_filter,
        )[0]

        self._update(obj, data, with_task)

        return obj

    def update(self, saver_filter_id, data, with_task=True):
        obj = self.get_qs(saver_filter_id).get()
        self._update(obj, data, with_task)

        return obj

    def _update(self, obj, data, with_task):
        for field in self.FIELDS:
            if field in data:
                setattr(obj, field, data[field])

        obj.save()

        if with_task and any(getattr(obj, f) for f in self.SUBSCRIPTION_FIELDS):
            self._run_task(obj)

    def _run_task(self, obj):
        method = UpdatePersonsFilters
        if yenv.type != 'development':
            method = method.delay
        method(filter_ids=[obj.persons_filter.id])

    def delete(self, saver_filter_id):
        obj = self.get_qs(saver_filter_id).get()
        obj.delete()


class PersonsFilterCtl(object):
    def __init__(self, persons_filter):
        self.persons_filter = persons_filter

    def update(self):
        persons_qs = (
            Staff.objects
            .filter(is_dismissed=False)
            .values_list('id', flat=True)
        )

        if self.persons_filter.filter_id:
            saved_filter = FilterCtl.get_by_id(self.persons_filter.filter_id)
            if not saved_filter.is_valid():
                logger.info(f'Skipping filter {self.persons_filter.filter_id} because it is not valid')
                return
            filter_q = saved_filter.get_query()
            persons_qs = persons_qs.filter(filter_q)

        if self.persons_filter.department:
            persons_qs = filter_by_heirarchy(
                persons_qs,
                [self.persons_filter.department],
                by_children=True,
                include_self=True,
                filter_prefix='department__',
            )

        persons_qs = persons_qs.order_by('id')

        person_ids = list(persons_qs)

        result_hash = ','.join(str(_id) for _id in person_ids)
        result_hash = hashlib.sha1(result_hash.encode('utf-8')).hexdigest()

        if self.persons_filter.result_hash != result_hash:
            self.persons_filter.result_hash = result_hash
            self.persons_filter.persons_count = len(person_ids)
            self.persons_filter.persons.clear()
            PersonToPersonFilter.objects.bulk_create(
                [
                    PersonToPersonFilter(
                        personsfilter=self.persons_filter,
                        staff_id=person_id,
                    ) for person_id in person_ids
                ],
            )
            self.persons_filter.updated_at = datetime.now()

        self.persons_filter.save()

    @classmethod
    def get_by_filter_id(cls, filter_id):
        persons_filter = PersonsFilter.objects.get(id=filter_id)
        return cls(persons_filter)

    @classmethod
    def get_update_candidates(cls):
        qs = (
            PersonsFilter.objects
            .filter(saved_filters__person__is_dismissed=False)
            .distinct()
        )
        return (cls(pf) for pf in qs)


def get_subscribers(person, subscription):
    """ Отбирает людей, у которых есть сохраненные фильтры
    с нужным типом подписки
    и связанные с фильтрами, которые содержат указанного человека."""
    assert subscription in SavedFilterCtl.SUBSCRIPTIONS
    subscription_query = {
        'saved_filters__%s_subscription' % subscription: True,
        'saved_filters__persons_filter__persons': person,
        'is_dismissed': False,
    }
    return Staff.objects.filter(**subscription_query).exclude(id=person).distinct()
