import json

from django import forms
from django.views.decorators.http import require_http_methods
from django.views.decorators.csrf import ensure_csrf_cookie, csrf_exempt
from django.http import HttpResponseBadRequest, HttpResponseNotFound
from django.core.exceptions import ObjectDoesNotExist

from staff.lib.decorators import responding_json, paginated, available_for_external
from staff.lib.utils.qs_values import localize

from staff.person.models import Staff

from .controller import FilterCtl, FilterNotFound
from .forms import SavedFilterForm, SavedFilterCreateForm
from .person_getter import FlatPersonsGetter
from .filter_context import FilterContext
from .saved_filter_ctl import SavedFilterCtl, get_subscribers


@ensure_csrf_cookie
@require_http_methods(['GET'])
@responding_json
def meta(request):
    return FilterCtl.as_meta_dict()


@require_http_methods(['POST'])
@responding_json
def create_filter(request):
    try:
        data = json.loads(request.read())
    except ValueError:
        return HttpResponseBadRequest('invalid_json')

    filter_ctl = FilterCtl.get_by_data(data=data)

    if filter_ctl.is_valid():
        return {'filter_id': filter_ctl.create_id()}
    else:
        return filter_ctl.get_errors()


# Операция недеструктивная, CSRF-проверка не нужна,
# POST — чтобы не передавать громоздкие фильтры в query-параметрах
@csrf_exempt
@require_http_methods(['POST'])
@responding_json
@paginated
def filter_persons(request, paginator):
    try:
        data = json.loads(request.read())
    except ValueError:
        return HttpResponseBadRequest('invalid_json')

    try:
        filter_context = FilterContext(filter_data=data)
    except FilterNotFound:
        return {'error': 'filter_not_found'}, 404

    person_getter = FlatPersonsGetter(
        filter_context=filter_context,
        page=paginator.page,
        limit=paginator.limit
    )

    paginator.result = person_getter.get_flat_persons()
    paginator.total = person_getter.count()
    return paginator


@ensure_csrf_cookie
@require_http_methods(['GET'])
@responding_json
def get_filter(request, filter_id):
    try:
        return FilterCtl.get_by_id(filter_id).as_dict()
    except FilterNotFound:
        return {'error': 'filter_not_found'}, 404


@ensure_csrf_cookie
@require_http_methods(['GET'])
@responding_json
@available_for_external
def saved_filters_list(request):
    login = request.GET.get('login')
    contains_login = request.GET.get('contains_login')
    is_bookmark = request.GET.get('is_bookmark')
    is_for_cab = request.GET.get('is_for_cab')
    absences_subscription = request.GET.get('absences_subscription')
    birthdays_subscription = request.GET.get('birthdays_subscription')

    if request.user.is_superuser and login:
        person = Staff.objects.get(login=login)
    else:
        person = request.user.get_profile()
    return SavedFilterCtl(person).get_list(
        contains_login=contains_login,
        is_bookmark=is_bookmark,
        is_for_cab=is_for_cab,
        absences_subscription=absences_subscription,
        birthdays_subscription=birthdays_subscription,
    )


@require_http_methods(['GET', 'POST'])
@responding_json
def saved_filter_create(request):
    saved_filter_ctl = SavedFilterCtl(request.user.get_profile())

    if request.method == 'GET':
        form = SavedFilterCreateForm()
        return form.as_dict()

    elif request.method == 'POST':
        form = SavedFilterCreateForm(data=json.loads(request.read()))

        if not form.is_valid():
            return form.errors

        saved_filter_obj = saved_filter_ctl.create(form.cleaned_data)
        return {'saved_filter': saved_filter_ctl.get(saved_filter_obj.id)}


@require_http_methods(['GET', 'POST', 'DELETE'])
@responding_json
def saved_filter(request, saved_filter_id):
    saved_filter_ctl = SavedFilterCtl(request.user.get_profile())

    try:
        if request.method == 'DELETE':
            saved_filter_ctl.delete(saved_filter_id)
            return {'success': True}

        elif request.method == 'GET':
            form = SavedFilterForm(
                initial=saved_filter_ctl.get(saved_filter_id)
            )
            return form.as_dict()

        elif request.method == 'POST':
            form = SavedFilterForm(
                initial=saved_filter_ctl.get(saved_filter_id),
                data=json.loads(request.read()),
            )

            if form.is_valid():
                saved_filter_obj = saved_filter_ctl.update(
                    saved_filter_id, form.cleaned_data
                )
                return {'saved_filter': saved_filter_ctl.get(saved_filter_obj.id)}
            else:
                return form.errors

    except ObjectDoesNotExist:
        return HttpResponseNotFound('Saved Filter not Found')


@require_http_methods(['GET'])
@responding_json
@paginated
def saved_filter_persons(request, saved_filter_id, paginator):
    saved_filter_ctl = SavedFilterCtl(request.user.get_profile())
    saved_filter_data = saved_filter_ctl.get(saved_filter_id)
    if saved_filter_data is None:
        return {'error': 'saved_filter_not_found'}, 404

    url = saved_filter_data.get('department_url')
    filter_id = saved_filter_data.get('filter_id')

    try:
        filter_context = FilterContext(filter_id=filter_id)
    except FilterNotFound:
        return {'error': 'filter_not_found'}, 404

    person_getter = FlatPersonsGetter(
        filter_context,
        url,
        page=paginator.page,
        limit=paginator.limit
    )

    paginator.result = person_getter.get_flat_persons()
    paginator.total = person_getter.count()
    return paginator


@require_http_methods(['GET'])
@available_for_external
@responding_json
def subscribers(request):
    subscription = request.GET.get('subscription', 'absences')
    if subscription not in SavedFilterCtl.SUBSCRIPTIONS:
        return HttpResponseBadRequest('Unknown subscription type.')

    login = request.GET.get('login')
    if request.user.is_superuser and login:
        person = Staff.objects.get(login=login)
    else:
        person = request.user.get_profile()

    return [
        localize(s) for s in
        get_subscribers(person.id, subscription)
        .values(
            'id',
            'login',
            'first_name',
            'last_name',
            'first_name_en',
            'last_name_en',
        )
    ]


@ensure_csrf_cookie
@require_http_methods(['GET', 'POST'])
@responding_json
def replace_filter(request):
    from staff.person.models import EMPLOYMENT

    DAYMONTH_INPUT_FORMATS = ('%d.%m', '%m/%d', '%d.%m.%Y', '%m/%d/%Y')
    date_dj_field = forms.DateField(input_formats=DAYMONTH_INPUT_FORMATS)

    data = []

    def query(field, condition='Equal', operator='and', **kwargs):
        result = locals()
        result.update(result.pop('kwargs'))
        return result

    def add(field, condition='Equal', operator='and', **kwargs):
        data.append(query(field, condition, operator, **kwargs))

    params = request.GET or request.POST

    if params.get('is_chief'):
        add('ChiefFilter', value=True)

    if params.get('employment'):
        add('EmploymentFilter', employment=EMPLOYMENT.get_name(params['employment']).lower())

    if params.get('organization'):
        add('OrganizationFilter', organization=params['organization'])

    if params.get('department'):
        add('DepartmentFilter', 'InHierarchy', department=params['department'])

    if params.get('country'):
        add('CountryFilter', country=params['country'])

    if params.get('city'):
        add('CityFilter', city=params['city'])

    if params.get('office'):
        add('OfficeFilter', office=params['office'])

    gender = {'m': 'male', 'f': 'female'}.get(params.get('gender', '').lower())
    if gender:
        add('GenderFilter', gender=gender)

    if params.get('family_status'):
        add('FamilyStatusFilter', value=params['family_status'].lower() == 'm')

    if params.get('tshirt_size'):
        add('TshirtSizeFilter', tshirt_size=params['tshirt_size'])

    if params.get('owns_a_car') in ('2', '3'):
        add('HasAutoFilter', value=str(params.get('owns_a_car')) == '2')

    if params.get('birth_year'):
        add('BirthdayFilter', 'DayMonthYear', year=params['birth_year'])

    try:
        if params.get('join_at_0') or params.get('join_at_1'):
            start = (
                date_dj_field.clean(params.get('join_at_0')).isoformat()
                if params.get('join_at_0') else ''
            )
            end = (
                date_dj_field.clean(params.get('join_at_1')).isoformat()
                if params.get('join_at_1') else ''
            )
            add('JoinAtFilter', 'Range', start=start, end=end)

        if params.get('birthday_0') or params.get('birthday_1'):
            start = (
                date_dj_field.clean(params.get('birthday_0')).isoformat()
                if params.get('birthday_0') else ''
            )
            end = (
                date_dj_field.clean(params.get('birthday_1')).isoformat()
                if params.get('birthday_0') else ''
            )
            add('BirthdayFilter', 'Range', start=start, end=end)
    except Exception:
        return HttpResponseBadRequest('invalid_filter')

    if params.get('qfilter') == 'new':
        add('BeginnerFilter', value=True)
    elif params.get('qfilter') == 'bosses' and not params.get('is_chief'):
        add('ChiefFilter', value=True)
    elif params.get('qfilter') == 'absent':
        add('GapFilter', 'Existence', value=True)

    if params.get('is_head') or params.get('qfilter') == 'heads':
        data = (
            [query('ChiefFilter', value=True)]
            + data
            + [query('DeputyFilter', operator='or', value=True)]
            + data
        )

    result = {}

    if data:
        filter_ctl = FilterCtl.get_by_data(data={'filters': data})
        if filter_ctl.is_valid():
            result = {'filter_id': filter_ctl.create_id()}
        else:
            return HttpResponseBadRequest('invalid_filter')

    unsupported_filds = [
        'is_dismissed',
        'quit_at',
        'car_model',
        'phone_model',
        'school',
        'graduation_year',
    ]
    for field in unsupported_filds:
        if params.get(field):
            result.setdefault('unsupported', []).append(field)

    if params.get('qfilter') == 'candidates':
        result.setdefault('unsupported', []).append('candidates')

    return result
