from typing import Any, Dict
from django.conf import settings
from django.utils.translation import ugettext as _
from django.utils.html import escape

from staff.person_profile.cards.utils import format_date_ago
from staff.person_profile.controllers.blocks.abstract_block import AbstractBlock


class CardPersonalBlock(AbstractBlock):
    name = 'personal'
    fields = {'gender'}

    def get_data(self) -> Dict[str, Any]:
        data = self.get_raw_data()
        return data


class CardNamesBlock(AbstractBlock):
    name = 'names'
    fields = {'first_name', 'last_name'}

    def get_data(self) -> Dict[str, Any]:
        data = self.get_raw_data()
        return {
            'first_name': escape(data['first_name']),
            'last_name': escape(data['last_name']),
        }


class CardPhonesBlock(AbstractBlock):
    name = 'phones'
    fields = {'work_phone', 'mobile_phone'}

    def get_data(self) -> Dict[str, Any]:
        data = self.get_raw_data()
        return {
            'phones': {
                'mobile': data['mobile_phone'] and data['mobile_phone'].split(',')[0] or '',
                'work': data['work_phone'],
            }
        }


class CardLocationOfficeBlock(AbstractBlock):
    name = 'location_office'
    fields = {'office', 'is_dismissed'}

    def get_data(self) -> Dict[str, Any]:
        data = self.get_raw_data()
        if data['is_dismissed']:
            return {}

        return {
            'office_id': data['office']['id'],
            'office': data['office']['name']
        }


class CardLocationRoomBlock(AbstractBlock):
    name = 'location_room'
    fields = {'room'}

    def get_data(self) -> Dict[str, Any]:
        data = self.get_raw_data()

        return {
            'room_id': data['room']['id'],
            'room': data['room']['name'],
        }


class CardLocationTableBlock(AbstractBlock):
    name = 'location_table'
    fields = {'table_id'}

    def get_data(self) -> Dict[str, Any]:
        data = self.get_raw_data()
        if not data['table_id']:
            return {}

        return {
            'table': {
                'number': data['table_id'],
                'url': '//{host}/map/table/{table_id}/'.format(
                    host=settings.STAFF_HOST,
                    table_id=data['table_id'],
                )
            }
        }


class CardBaseFieldsBlock(AbstractBlock):
    name = 'base_fields'
    fields = {'login', 'work_email', 'custom_work_email'}

    def get_data(self) -> Dict[str, Any]:
        data = self.get_raw_data()
        data['staff_url'] = '//{host}/{login}'.format(
            host=settings.STAFF_HOST,
            login=data['login'],
        )
        data['avatar'] = '//{host}/user/avatar/{login}/100/square'.format(
            host=settings.CENTER_MASTER,
            login=data['login'],
        )
        return data


class CardAvatarFieldsBlock(AbstractBlock):
    name = 'avatar'
    fields = {'login'}

    def get_data(self) -> Dict[str, Any]:
        data = self.get_raw_data()
        data['avatar'] = '//{host}/user/avatar/{login}/100/square'.format(
            host=settings.CENTER_MASTER,
            login=data['login'],
        )
        return data


class CardDepartmentFieldsBlock(AbstractBlock):
    fields = {'department', 'memorial_profile__intranet_status', 'is_dismissed'}

    def get_data(self) -> Dict[str, Any]:
        data = self.get_raw_data()
        if data['memorial_profile__intranet_status']:
            return {'dep': {'name': '', 'bg_color': '#909090', 'fg_color': '#ffffff'}}

        if data['is_dismissed']:
            return {
                'dep': {
                    'name': _('api.StaffCards_FormerEmployee'),
                    'bg_color': '#909090',
                    'fg_color': '#ffffff',
                }
            }

        return {
            'dep': {
                'name': data['department']['name'],
                'bg_color': data['department']['bg_color'],
                'fg_color': data['department']['fg_color'],
            }
        }


class CardLastOfficeBlock(AbstractBlock):
    name = 'last_office'
    fields = {'last_activity'}

    def get_data(self) -> Dict[str, Any]:
        data = self.get_raw_data()
        if not data['last_activity']:
            return {'jabber_status': 'offline'}

        return {
            'jabber_status': 'offline',
            'last_office': {
                'ago': format_date_ago(data['last_activity']['updated_at'], None, False),
                'office': data['last_activity']['office'],
            },
        }


class CardGapBlock(AbstractBlock):
    name = 'gap'
    fields = {'gap'}

    def _extract_gap_time(self, gap: dict) -> str:
        if gap['left_edge'] == gap['right_edge']:
            if gap['full_day']:
                return _('api.StaffCards_Today')

            return ''

        return '%s %s&nbsp;%s' % (
            _('api.StaffCards_Till'),
            gap['right_edge'].day,
            _(gap['right_edge'].strftime('%B') + 'z'),
        )

    def get_data(self) -> Dict[str, Any]:
        data = self.get_raw_data()
        if not data['gap']:
            return {}

        return {
            'gap': {
                'url': '//{host}/'.format(host=settings.GAP_HOST),
                'color': data['gap']['color'],
                'name': self.get_gap_name(data['gap']),
                'time': self._extract_gap_time(data['gap']),
            }
        }

    def get_gap_name(self, gap):
        names = {
            'absence': _('api.StaffCards_IsAbsent'),
            'remote_work': _('api.StaffCards_IsWorkingRemotely'),
            'office_work': _('api.StaffCards_IsWorkingInOffice'),
            'duty': _('api.StaffCards_IsDuty'),
            'vacation': _('api.StaffCards_IsOnVacation'),
            'paid_day_off': _('api.StaffCards_IsOnPaidDayOff'),
            'selfpaid': _('api.StaffCards_IsOnVacation'),
            'illness': _('api.StaffCards_IsIll'),
            'illness_covid': _('api.StaffCards_IsIll_Covid'),
            'trip': _('api.StaffCards_IsAwayOnBusiness'),
            'learning': _('api.StaffCards_IsOnLearning'),
            'maternity': _('api.StaffCards_IsOnParentalLeave'),
            'conference_trip': _('api.StaffCards_IsOnConferenceTrip'),
        }

        key = gap['name']
        if gap['name'] == 'illness' and gap['is_covid']:
            key = 'illness_covid'

        return names.get(key, gap['name'])


class CardPositionBlock(AbstractBlock):
    name = 'position'
    fields = {'position', 'office', 'is_dismissed', 'memorial_profile__intranet_status'}

    def get_data(self) -> Dict[str, Any]:
        data = self.get_raw_data()
        return {
            'is_dismissed': data['is_dismissed'],
            'is_homie': data['office']['id'] == settings.HOMIE_OFFICE_ID,
            'is_memorial': bool(data['memorial_profile__intranet_status']),
            'position': escape(data['position']),
        }


class CardAccountsBlock(AbstractBlock):
    name = 'accounts'
    fields = {'login'}

    def get_data(self) -> Dict[str, Any]:
        data = self.get_raw_data()
        login = data['login']
        return {
            'accounts': [
                {
                    'name': _('api.StaffCards_Wiki'),
                    'url': '//{host}/~{login}'.format(host=settings.WIKI_HOST, login=login),
                },
                {
                    'name': _('api.StaffCards_Diary'),
                    'url': '//{login}.{host}'.format(host=settings.ATUSHKA_HOST, login=login),
                },
                {
                    'name': _('api.StaffCards_Meetings'),
                    'url': '//{host}/invite/my/{login}'.format(
                        host=settings.CALENDAR_HOST, login=login
                    ),
                },
            ]
        }
