from datetime import datetime, timedelta
from django.utils.translation import gettext_lazy, ngettext_lazy, gettext, ngettext, string_concat


_DATETIME_FORMATS = (
    '%Y-%m-%d %H:%M:%S',
    '%Y-%m-%d %H:%M',
    '%Y-%m-%dT%H:%M:%S',
    '%Y-%m-%dT%H:%M',
    '%Y-%m-%d',
)

TD_TWO_MINUTES = timedelta(minutes=2)
TD_ONE_HOUR = timedelta(hours=1)
TD_TWO_HOURS = timedelta(hours=2)
TD_ONE_WEEK = timedelta(weeks=1)


def format_date_ago(dt, limit=None, lazy=True):
    now = datetime.now()
    ago = now - dt
    gt = gettext_lazy if lazy else gettext
    ngt = ngettext_lazy if lazy else ngettext

    if limit and ago > limit:
        return None

    if ago < TD_TWO_MINUTES:
        return gt('api.date_ago_Just')

    if ago < TD_ONE_HOUR:
        if lazy:
            return string_concat(
                int(ago.seconds // 60),
                ' ',
                ngettext_lazy(
                    'api.date_ago_xMinuteAgo', 'api.date_ago_xMinutesAgo', ago.seconds / 60
                ),
            )
        else:
            return (
                str(int(ago.seconds // 60))
                + ' '
                + ngt('api.date_ago_xMinuteAgo', 'api.date_ago_xMinutesAgo', ago.seconds / 60)
            )

    if ago < TD_TWO_HOURS:
        return gt('api.date_ago_MoreThanHourAgo')

    today = timedelta(hours=now.hour, minutes=now.minute, seconds=now.second)
    if ago < today:
        if lazy:
            return string_concat(
                int(ago.seconds // 3600),
                ' ',
                ngettext_lazy(
                    'api.date_ago_xHourAgo', 'api.date_ago_xHoursAgo', ago.seconds / 3600
                ),
            )
        else:
            return (
                str(int(ago.seconds // 3600))
                + ' '
                + ngt('api.date_ago_xHourAgo', 'api.date_ago_xHoursAgo', ago.seconds / 3600)
            )

    yesterday = today + timedelta(days=1)
    if ago < yesterday:
        return gt('api.date_ago_Yesterday')

    day_before = yesterday + timedelta(days=1)
    if ago < day_before:
        return gt('api.date_ago_DayBefore')

    if ago < TD_ONE_WEEK:
        if lazy:
            return string_concat(
                ago.days,
                ' ',
                ngettext_lazy('api.date_ago_DayAgo', 'api.date_ago_DaysAgo', ago.days),
            )
        else:
            return (
                str(ago.days) + ' ' + ngt('api.date_ago_DayAgo', 'api.date_ago_DaysAgo', ago.days)
            )

    return gt('api.date_ago_AgesAgo')
