import logging
from calendar import monthrange
from datetime import date, timedelta, datetime
from decimal import Decimal, ROUND_DOWN
from typing import AnyStr, Optional, List, Tuple, Union, Any

import attr
from django.conf import settings

from staff.lib import (
    requests,
    attr_ext,
)
from staff.lib.utils.ordered_choices import OrderedChoices
from staff.lib import tvm2
from staff.person.models import Staff
from staff.lib.calendar import get_holidays

logger = logging.getLogger(__name__)

TWO = Decimal('0.01')


PAY_TYPES = OrderedChoices(
    ('DAY', 'Day'),
    ('MONTH', 'Month')
)
FOOD_ACCOUNT = 'Food'
BREAKFAST_ACCOUNT = 'Breakfast'
SNAK_ACCOUNT = 'Snack'
RUSSIA_GEO_ID = 225


def _datetime_converter(value):
    if not isinstance(value, datetime):
        return datetime.strptime(value[:len('2017-01-01T23:03:03')], '%Y-%m-%dT%H:%M:%S')
    return value


class BadgepayAnswerError(Exception):
    pass


@attr.s
class Account(object):
    account = attr.ib(type=AnyStr)
    sum = attr.ib(type=Decimal, converter=Decimal)
    ptype = attr.ib(default=None, type=AnyStr)
    days = attr.ib(default=None, type=int)


@attr.s
class Order(object):

    @attr.s
    class Cashflow(object):
        account = attr.ib(type=AnyStr)
        sum = attr.ib(type=Decimal, converter=Decimal)

    date = attr.ib(type=datetime, converter=_datetime_converter)
    merchant = attr.ib(type=str)
    sum = attr.ib(type=Decimal, converter=Decimal)
    cashflows = attr.ib(type=List[Cashflow], converter=attr_ext.list_of(Cashflow), factory=list)


def _is_food(obj: Union[Order.Cashflow, Account]) -> bool:
    return obj.account == FOOD_ACCOUNT


def _get_pay_type(obj: Union[Order.Cashflow, Account]) -> Optional[str]:
    pay_type = getattr(obj, 'ptype', None)
    return pay_type if pay_type in PAY_TYPES else None


@attr.s
class BadgepayReport(object):
    benefits = attr.ib(type=List[Account], converter=attr_ext.list_of(Account))
    limits = attr.ib(type=List[Account], converter=attr_ext.list_of(Account))
    expenses = attr.ib(type=List[Account], converter=attr_ext.list_of(Account))
    orders = attr.ib(type=List[Order], converter=attr_ext.list_of(Order))
    login = attr.ib(type=str)
    overrun = attr.ib(type=Decimal, converter=Decimal)

    def _get_food_sum(self, section: str, pay_type: PAY_TYPES = None) -> Decimal:
        return self._get_food_account_value(section, field='sum', pay_type=pay_type, default=Decimal(0))

    def _get_food_days(self, section: str, pay_type: PAY_TYPES = None) -> int:
        return self._get_food_account_value(section, field='days', pay_type=pay_type)

    def _get_food_account_value(self, section: str, field: str, pay_type: PAY_TYPES = None, default=None) -> Any:
        food_value = (
            getattr(it, field) for it in filter(
                lambda s: _is_food(s) and _get_pay_type(s) == pay_type,
                getattr(self, section)
            )
        )
        return next(food_value, default)

    @property
    def food_month_sum(self) -> Decimal:
        return self._get_food_sum('benefits', pay_type=PAY_TYPES.MONTH)

    @property
    def food_day_sum(self) -> Decimal:
        return self._get_food_sum('benefits', pay_type=PAY_TYPES.DAY)

    @property
    def food_spent_sum_month(self) -> Decimal:
        """Трата за месяц"""
        return self._get_food_sum('expenses', pay_type=PAY_TYPES.MONTH)

    @property
    def food_spent_sum_day(self) -> Decimal:
        """Трата за день"""
        return self._get_food_sum('expenses', pay_type=PAY_TYPES.DAY)

    @property
    def food_month_remained_sum(self) -> Decimal:
        """Остаток до конца месяца"""
        return self.food_month_sum - self.food_spent_sum_month

    @property
    def food_day_remained_sum(self) -> Decimal:
        """Остаток до конца дня"""

        # Если все дни были потрачены, то вычитать надо из нуля
        if self.days_limit - self.days_used <= 0:
            return -self.food_spent_sum_day

        return self.food_day_sum - self.food_spent_sum_day

    @property
    def days_limit(self) -> int:
        return self._get_food_days('benefits', pay_type=PAY_TYPES.DAY) or 0

    @property
    def days_used(self) -> int:
        return (
            self._get_food_days('expenses', pay_type=PAY_TYPES.DAY)
            or self._get_food_days('expenses', pay_type=PAY_TYPES.MONTH)
            or 0
        )


def get_badgepay_report(login: str, limit: int = 1) -> BadgepayReport:
    response = requests.post(
        url='https://{host}/pg/integration/payers/monthStat'.format(host=settings.BADGEPAY_HOST),
        json={'login': login, 'orderlimit': limit},
        headers={tvm2.TVM_SERVICE_TICKET_HEADER: tvm2.get_tvm_ticket_by_deploy('badgepay')},
        timeout=(1, 2, 5),
    )
    if response.status_code != 200:
        raise BadgepayAnswerError('Badgepy answers {}'.format(response.status_code))
    response = response.json()
    if not isinstance(response, dict):
        raise TypeError('Badgepy returned wrong json with type {}'.format(response.__class__))
    return attr_ext.from_kwargs(BadgepayReport, **response)


@attr.s
class ShortFoodReport(object):
    work_days_left = attr.ib(type=int)
    recommended_per_day = attr.ib(type=Decimal)
    food_balance_compensation_days = attr.ib(type=int)
    food_days_total = attr.ib(type=int)
    month_remained_sum = attr.ib(type=Decimal)
    day_remained_sum = attr.ib(type=Decimal)
    month_limit = attr.ib(type=Decimal)
    day_limit = attr.ib(type=Decimal)
    overspending = attr.ib(type=Decimal)


def get_short_food_report(login):
    # type: (AnyStr) -> Optional[ShortFoodReport]
    try:
        report = get_badgepay_report(login)
    except Exception as exc:
        logger.warning('Couldn\'t receive badgepay report due to: {}'.format(exc))
        return
    work_days_left, _ = _get_remaining_and_passed_work_days(login)

    food_days_left = report.days_limit - report.days_used

    # Если баланс дня (сумма) <= 0, то количество дней компенсации ("days" : 3) нужно уменьшать на 1.
    daily_overrun_penalty = 1 if food_days_left and report.food_day_remained_sum < 0 else 0

    return ShortFoodReport(
        work_days_left=work_days_left,
        food_balance_compensation_days=food_days_left - daily_overrun_penalty,
        food_days_total=report.days_limit,
        recommended_per_day=_calc_sum_per_day(report.food_month_remained_sum, work_days_left),
        month_remained_sum=report.food_month_remained_sum,
        day_remained_sum=report.food_day_remained_sum,
        month_limit=report.food_month_sum,
        day_limit=report.food_day_sum,
        overspending=report.overrun,
    )


@attr.s
class ReportOrder(object):
    date = attr.ib(type=datetime)
    merchant = attr.ib(type=str)
    sum = attr.ib(type=Decimal)
    food = attr.ib(type=Decimal)
    breakfast = attr.ib(type=Decimal)
    snack = attr.ib(type=Decimal)
    money = attr.ib(type=Decimal, converter=Decimal)


@attr.s
class FullFoodReport(object):
    work_days_passed = attr.ib(type=int)
    work_days_left = attr.ib(type=int)

    food_balance_compensation_days = attr.ib(type=int)
    food_days_total = attr.ib(type=int)
    recommended_per_day = attr.ib(type=Decimal)
    middle_spent_per_day = attr.ib(type=Decimal)
    remained_sum_month = attr.ib(type=Decimal)
    remained_sum_day = attr.ib(type=Decimal)
    limit_month = attr.ib(type=Decimal)
    limit_day = attr.ib(type=Decimal)
    overspending = attr.ib(type=Decimal)

    orders = attr.ib(type=List[ReportOrder])


def get_full_food_report(login: str) -> Optional[FullFoodReport]:
    try:
        report = get_badgepay_report(login, 1000)  # limit is random, feel free to change if needed
    except Exception as exc:
        logger.warning('Couldn\'t receive badgepay report due to: {}'.format(exc))
        return
    work_days_left, work_days_passed = _get_remaining_and_passed_work_days(login)

    food_days_left = report.days_limit - report.days_used

    # Если баланс дня (сумма) <= 0, то количество дней компенсации ("days" : 3) нужно уменьшать на 1.
    daily_overrun_penalty = 1 if food_days_left and report.food_day_remained_sum < 0 else 0

    return FullFoodReport(
        work_days_left=work_days_left,
        work_days_passed=work_days_passed,

        food_balance_compensation_days=food_days_left - daily_overrun_penalty,
        food_days_total=report.days_limit,
        recommended_per_day=_calc_sum_per_day(report.food_month_remained_sum, work_days_left),
        middle_spent_per_day=_calc_sum_per_day(report.food_spent_sum_month, work_days_passed),
        remained_sum_month=report.food_month_remained_sum,
        remained_sum_day=report.food_day_remained_sum,
        limit_month=report.food_month_sum,
        limit_day=report.food_day_sum,
        overspending=report.overrun,

        orders=_create_orders_for_report(report),
    )


def get_today():
    """ is needed to patch in tests """
    return date.today()


def _create_orders_for_report(badgepay_report: BadgepayReport) -> List[ReportOrder]:
    res = []
    for order in badgepay_report.orders:
        food, breakfast, snack = Decimal(0), Decimal(0), Decimal(0)
        for flow in order.cashflows:
            if flow.account == BREAKFAST_ACCOUNT:
                breakfast += flow.sum
            elif flow.account == FOOD_ACCOUNT:
                food += flow.sum
            elif flow.account == SNAK_ACCOUNT:
                snack += flow.sum
        res.append(ReportOrder(
            merchant=order.merchant,
            date=order.date,
            sum=order.sum,
            breakfast=breakfast,
            food=food,
            snack=snack,
            money=Decimal(0),
        ))

    return res


def _get_remaining_and_passed_work_days(login):
    # type: (AnyStr) -> Tuple[int, int]
    geo_id = (
        Staff.objects
        .values_list('office__city__country__geo_base_id')
        .filter(login=login)
        .first()
    ) or RUSSIA_GEO_ID
    today = get_today()
    month = Month(today)
    holidays = month.holidays(geo_id)
    days_off = len([h for h in holidays if h['date'] >= today])
    remaining_delta = month.last_day - today + timedelta(days=1)
    remaining_days = remaining_delta.days - days_off
    passed_days = month.days_count - len(holidays) - remaining_days
    return remaining_days, passed_days


def _calc_sum_per_day(remained_sum, num_of_days):
    # type: (Decimal, int) -> Decimal
    if remained_sum < 0:
        sum_per_day = Decimal(0)
    elif num_of_days:
        sum_per_day = Decimal(remained_sum / num_of_days)
    else:
        sum_per_day = Decimal(remained_sum)
    return sum_per_day.quantize(TWO, ROUND_DOWN)


class Month(tuple):

    def __new__(cls, day):

        days_count = monthrange(day.year, day.month)[1]
        first_day = date(year=day.year, month=day.month, day=1)
        last_day = date(year=day.year, month=day.month, day=days_count)

        month = super(Month, cls).__new__(
            cls,
            (first_day + timedelta(i) for i in range(days_count))
        )

        month.first_day = first_day
        month.last_day = last_day
        month.days_count = days_count
        month._holidays = {}

        return month

    def _load_holidays(self, country_geo_id):
        return get_holidays(
            self.first_day,
            self.last_day,
            [country_geo_id],
            out_mode='holidays',
        ).get(country_geo_id)

    def holidays(self, geo_id=225):
        if geo_id not in self._holidays:
            self._holidays[geo_id] = self._load_holidays(geo_id)

        return self._holidays[geo_id]
