from typing import List, Dict, Optional, Iterable

from staff.lib.utils.qs_values import extract_related, localize
from staff.lib.models.roles_chain import get_grouped_roles_chains_by_departments, DepListType, DepDataType

from staff.departments.utils import get_department_chain
from staff.departments.models import DepartmentRoles, ValuestreamRoles

from staff.person_profile.controllers.blocks.abstract_block import AbstractBlock
from staff.person_profile.controllers.blocks.utils import get_person_fields


def get_roles_data(department_list: DepListType) -> Dict:
    roles = (
        DepartmentRoles.CHIEF.value,
        DepartmentRoles.GENERAL_DIRECTOR.value,
        DepartmentRoles.HR_PARTNER.value,
        ValuestreamRoles.HEAD.value,
    )

    roles_by_departments = get_grouped_roles_chains_by_departments(
        department_list=department_list,
        roles=roles,
        fields=get_person_fields(),
    )
    return roles_by_departments


class AbstractDepChainBlock(AbstractBlock):
    fields = {
        'id',
        'login',
        'department__lft',
        'department__rght',
        'department__tree_id',
        'department__id',
    }

    department_roles: Optional[Iterable[str]] = None

    def get_dep_data(self) -> DepDataType:
        return extract_related(self.data, 'department', pop=False)

    def get_department_chain(self):
        dep_data = self.get_dep_data()

        if not dep_data:
            return []

        department_id = dep_data['id']

        if department_id not in self.cache.get('department_chain', {}):
            dep_chain = get_department_chain(**dep_data)
            self.cache.setdefault('department_chain', {})[department_id] = [localize(d) for d in dep_chain]

        return self.cache['department_chain'][department_id]

    def get_roles_chain(self) -> List[List[Dict]]:
        dep_data = self.get_dep_data()

        if not dep_data:
            return []

        department_id = dep_data['id']

        if department_id not in self.cache.get('roles_chain', {}):
            self.cache.setdefault('roles_chain', {}).update(get_roles_data([dep_data]))

        return self.cache['roles_chain'][department_id]
