from typing import List, Dict, Set

from staff.lib.utils.qs_values import localize
from staff.person_profile.controllers.blocks.abstract_block import AbstractBlock
from staff.person_profile.controllers.blocks.birthday_block import BirthdayBlock
from staff.person_profile.controllers.blocks.dismissed_block import DismissedBlock, ShortDismissedBlock
from staff.person_profile.controllers.blocks.duties_block import DutiesBlock
from staff.person_profile.controllers.blocks.edu_block import EduBlock
from staff.person_profile.controllers.blocks.external_login_block import ExternalLoginBlock
from staff.person_profile.controllers.blocks.location_office_block import LocationOfficeBlock
from staff.person_profile.controllers.blocks.location_room_block import LocationRoomBlock
from staff.person_profile.controllers.blocks.location_table_block import LocationTableBlock
from staff.person_profile.controllers.blocks.memorial_block import MemorialBlock
from staff.person_profile.controllers.blocks.name_block import NameBlock
from staff.person_profile.controllers.blocks.official_organization_block import OfficialOrganizationBlock
from staff.person_profile.controllers.blocks.official_base_block import OfficialBaseBlock
from staff.person_profile.controllers.blocks.paid_day_off_block import PaidDayOffBlock
from staff.person_profile.controllers.blocks.password_block import PasswordBlock
from staff.person_profile.controllers.blocks.personal_block import PersonalBlock
from staff.person_profile.controllers.blocks.properties_block import PropertiesBlock
from staff.person_profile.controllers.blocks.city_block import CityBlock
from staff.person_profile.controllers.blocks.vacation_block import VacationBlock
from staff.person_profile.controllers.blocks.work_mode_block import WorkModeBlock
from staff.person_profile.controllers.blocks.work_phone_block import WorkPhoneBlock
from staff.person_profile.permissions.base import BaseRegistry


class MetaBlocksRegistry(object):
    _all_blocks = None

    @classmethod
    def get_all_blocks(cls) -> List[type]:
        return [
            BirthdayBlock,
            DismissedBlock,
            DutiesBlock,
            EduBlock,
            ExternalLoginBlock,
            LocationOfficeBlock,
            LocationRoomBlock,
            LocationTableBlock,
            MemorialBlock,
            NameBlock,
            OfficialOrganizationBlock,
            OfficialBaseBlock,
            CityBlock,
            PaidDayOffBlock,
            PasswordBlock,
            PersonalBlock,
            PropertiesBlock,
            ShortDismissedBlock,
            VacationBlock,
            WorkModeBlock,
            WorkPhoneBlock,
        ]

    @classmethod
    def get_fields(cls) -> Set[str]:
        fields = set()
        for block in cls.get_all_blocks():
            fields |= block.fields

        return fields

    def __init__(self, data: Dict, registry: BaseRegistry):
        self.registry = registry
        self.data = localize(data)
        self._inited_blocks = self._get_inited_blocks()

    def _get_available_blocks(self) -> List[type]:
        return [
            b for b in self.get_all_blocks()
            if b.name is None or
               b.name and self.registry.is_available(b.name, target_login=self.data['login'])
        ]

    def _get_inited_blocks(self) -> List[AbstractBlock]:
        cache = {}
        return [b(self.data, cache) for b in self._get_available_blocks()]

    def get_data(self):
        result = {}
        for block in self._inited_blocks:
            result.update(block.get_data())
        return result
