from typing import Any, Dict, List, Set

from staff.budget_position.models import BudgetPositionAssignment
from staff.groups.models import Group
from staff.lib.models.roles_chain import DepDataType
from staff.lib.utils.qs_values import extract_related

from staff.person_profile.controllers.blocks.abstract_dep_chain_block import AbstractDepChainBlock


def get_value_stream_data(value_stream_urls: Set[str]) -> Dict[str, Any]:
    group_data_qs = (
        Group.objects
        .filter(url__in=value_stream_urls)
        .values('url', 'service_id', 'service_tags')
    )

    return {item['url']: item for item in group_data_qs}


def get_persons_value_streams(logins: List[str]) -> Dict[str, Any]:
    value_stream_fields = {
        'person__login',
        'value_stream__lft',
        'value_stream__rght',
        'value_stream__tree_id',
        'value_stream__id',
        'value_stream__url',
    }

    main_assignments = (
        BudgetPositionAssignment.objects
        .filter(main_assignment=True, person__login__in=logins)
        .values(*value_stream_fields)
    )

    result = {
        main_assignment['person__login']: extract_related(main_assignment, 'value_stream', pop=False)
        for main_assignment in main_assignments
    }

    missing_logins = logins - result.keys()
    result.update(
        {login: {} for login in missing_logins},
    )

    return result


class ValueStreamBlock(AbstractDepChainBlock):
    def get_data(self):
        department_chain = self.get_department_chain()
        value_stream_urls = {x['url'] for x in department_chain}
        group_mapping = self._get_value_stream_data(value_stream_urls)

        for record in department_chain:
            group_data = group_mapping.get(record['url'], {})
            record['abc_service_id'] = group_data.get('service_id')
            record['service_tags'] = [x for x in group_data.get('service_tags', '').split(',') if x]

        has_vs = any('vs' in value_stream['service_tags'] for value_stream in department_chain)
        if not has_vs:
            return {'value_streams': []}

        return {'value_streams': department_chain}

    def _get_value_stream_data(self, value_stream_urls: Set[str]) -> Dict[str, Any]:
        missing_urls = value_stream_urls - self.cache.get('value_stream_data', {}).keys()

        if missing_urls:
            self.cache.setdefault('value_stream_data', {}).update(get_value_stream_data(missing_urls))

        return self.cache.get('value_stream_data', {})

    def get_dep_data(self) -> DepDataType:
        login = self.data['login']

        if login not in self.cache.get('person_value_stream_data', {}):
            result = get_persons_value_streams([login])
            self.cache.setdefault('person_value_stream_data', {}).update(result)

        return self.cache['person_value_stream_data'][login]
