import pytz
from datetime import timedelta
from collections import OrderedDict
from typing import Dict, Any, Optional

from staff.lib import tvm2

from staff.lib import requests

from django.conf import settings

from staff.map.models import Room, Office

from staff.lib.utils.qs_values import localize
from staff.lib.utils.date import parse_datetime

import logging
logger = logging.getLogger('person_profile.controllers.calendar')


UTC_TZ = pytz.timezone('UTC')
CALENDAR_ROUND_TIME = 15  # minutes
CALENDAR_CONNECTION_TIMEOUT = (0.2, 0.5, 1)  # seconds.


class CalendarError(Exception):
    pass


_calendar_session = requests.Session()


def _ask_calendar(handle: str, params: Dict[str, Any], observer_tvm_ticket: Optional[str] = None) -> requests.Response:

    url = '{protocol}://{host}/internal/{handle}'.format(
        handle=handle,
        protocol=settings.CALENDAR_API_PROTOCOL,
        host=settings.CALENDAR_API_HOST
    )
    user_ticket = observer_tvm_ticket or tvm2.get_user_ticket_for_robot_staff()
    try:
        response = _calendar_session.get(
            url,
            params=params,
            timeout=CALENDAR_CONNECTION_TIMEOUT,
            headers={
                tvm2.TVM_SERVICE_TICKET_HEADER: tvm2.get_tvm_ticket_by_deploy('calendar'),
                tvm2.TVM_USER_TICKET_HEADER: user_ticket,
            }
        )
    except requests.Timeout:
        logger.warning(
            'Connection to calendar timed out. Timeout is %s sec. Params: %s',
            CALENDAR_CONNECTION_TIMEOUT,
            params,
        )
        raise CalendarError

    return response


def event_as_dict(event, date_from, date_to, conference_rooms):
    full_day = event.get('isAllDay', False)

    if full_day:
        date_from = date_from.replace(tzinfo=None)
        date_to = date_to.replace(tzinfo=None) - timedelta(1)
    else:
        date_from = date_from.astimezone(UTC_TZ).replace(tzinfo=None)
        date_to = date_to.astimezone(UTC_TZ).replace(tzinfo=None)

    return {
        'type': 'meeting',
        'from': date_from,
        'to': date_to,
        'fullDay': full_day,
        'meta': {
            'id': event.get('eventId'),
            'name': event.get('eventName', ''),
            'conference_rooms': conference_rooms
        }
    }


def hydrate_events(events):
    if isinstance(events, dict):
        events = [events]

    rooms_names = get_names_from_rooms(collect_rooms_from_events(events))

    for event in events:
        date_from = parse_datetime(event.get('start'))
        date_to = parse_datetime(event.get('end'))

        if not date_from or not date_to:
            logger.error('Error parsing datetime from calendar for %s', event)
            continue

        conference_rooms = []
        for room in event.get('resources', []):
            try:
                room = rooms_names[room['email'].split('@')[0]]
            except (IndexError, KeyError):
                pass
            else:
                conference_rooms.append(room)

        yield event_as_dict(
            event=event,
            date_from=date_from,
            date_to=date_to,
            conference_rooms=conference_rooms
        )


def fetch_events_from_calendar(observer_uid: str, observer_tvm_ticket: str, target_email: str, date_from, date_to):
    params = {
        'from': date_from.strftime('%Y-%m-%dT%H:%M:%S%z'),
        'to': date_to.strftime('%Y-%m-%dT%H:%M:%S%z'),
        'uid': observer_uid,
        'emails': target_email,
        'display': 'events',
        'dateFormat': 'zoned',
        'shape': 'omit-participants',
    }

    response = _ask_calendar(
        handle='get-availability-intervals',
        params=params,
        observer_tvm_ticket=observer_tvm_ticket,
    )

    try:
        events = response.json()['subjectAvailabilities'][0]['intervals']
        return events
    except (KeyError, TypeError, ValueError):
        logger.info('Cannot parse calendar answer: %s', response.text, exc_info=True)
        raise CalendarError


def collect_rooms_from_events(events):
    rooms = set()
    for event in events:
        for room in event['resources']:
            rooms.add(room['email'].split('@')[0])
    rooms.discard(None)
    rooms.discard('')
    return rooms


def get_names_from_rooms(rooms_exchange_ids):
    rooms = (
        Room.objects
        .values('id', 'name', 'name_en', 'name_exchange', 'phone', 'video_conferencing', 'floor__office__code')
        .filter(name_exchange__in=rooms_exchange_ids)
    )

    result = {}
    for room in rooms:
        room = localize(room)
        phone = room.pop('phone') or ''
        video = room.pop('video_conferencing') or ''
        room['office_code'] = room.pop('floor__office__code', None)

        room['voice_phone'] = phone.strip()
        room['video_phone'] = video.strip()
        result[room['name_exchange'].lower()] = room

    return result


class PersonHolidays(object):

    def __init__(self, person, first_day, last_day):
        self.person = person
        self.first_day = first_day
        self.last_day = last_day

    def _get_city_geo_id(self):
        geo_id = (
            Office.objects
            .values_list('city__geo_id', flat=True)
            .get(id=self.person.office_id)
        )
        if not geo_id:
            geo_id = 225

        return geo_id

    @property
    def _get_params(self):
        params = {
            'from': self.first_day.isoformat(),
            'to': self.last_day.isoformat(),
            'for': self._get_city_geo_id(),
        }
        return params

    def _get_holidays(self):
        response = _ask_calendar(handle='get-holidays', params=self._get_params)
        try:
            holidays = response.json()['holidays']
            return holidays
        except (KeyError, TypeError, ValueError):
            logger.info('Cannot parse calendar answer: %s', response.text, exc_info=True)
            raise CalendarError

    def get_as_dict(self):
        result = OrderedDict()
        for day in self._get_holidays():
            result[day['date']] = {'type': day['type'], 'description': day.get('name', '')}
        return result
