import pymongo

from datetime import timedelta

from staff.gap.controllers.gap import (
    GapCtl,
    GapQueryBuilder,
)
from staff.gap.views.gaps_by_day_views import periodic_gaps_by_ids
from staff.gap.workflows.choices import LIMITED_STATES
from staff.gap.utils import show_delete_button, show_edit_button
from staff.map.models import Room, Office


def _enrich_gaps(gaps, office_ids, room_ids, periodic_gap_ids):
    offices = Office.objects.filter(id__in=office_ids).values('id', 'name', 'name_en', 'code')
    rooms = Room.objects.filter(id__in=room_ids).values('id', 'name', 'name_en', 'num')

    id_to_office = {o['id']: o for o in offices}
    id_to_room = {r['id']: r for r in rooms}
    id_to_periodic_gap = {
        periodic_gap.pop('id'): periodic_gap
        for periodic_gap in periodic_gaps_by_ids(list(periodic_gap_ids))
        if periodic_gap
    }

    for gap in gaps:
        periodic_gap_id = gap['meta'].get('periodic_gap_id')
        periodic_gap = id_to_periodic_gap.get(periodic_gap_id)
        if periodic_gap is not None:
            gap['meta'].update(periodic_gap)

        room_id = gap['meta'].get('room')
        room = id_to_room.get(room_id)
        if room is not None:
            gap['meta']['room_name'] = room['name']
            gap['meta']['room_name_en'] = room['name_en']
            gap['meta']['room_num'] = room['num']
        office_id = gap['meta'].get('office')
        office = id_to_office.get(office_id)
        if office is not None:
            gap['meta']['office_code'] = office['code']
            gap['meta']['office_name'] = office['name']
            gap['meta']['office_name_en'] = office['name_en']


def get_calendar_gaps(date_from, date_to, login, observer):
    gqb = (
        GapQueryBuilder()
        .person_login(login)
        .dates_not_strict(date_from, date_to)
        .workflows_states(LIMITED_STATES)
    )

    fields = [
        'id',
        'workflow',
        'date_from',
        'date_to',
        'full_day',
        'work_in_absence',
        'is_covid',
        'comment',
        'service_slug',
        'service_name',
        'shift_id',
        'role_on_duty',
        'place',
        'office',
        'room',
        'table',
        'periodic_gap_id',
        'mandatory',
        'vacation_updated',
        'deadline',
    ]

    sorting = [('date_from', pymongo.ASCENDING)]

    gaps = GapCtl().find_gaps(query=gqb.query(), fields=fields, sorting=sorting)

    res = []
    room_ids = set()
    office_ids = set()
    periodic_gap_ids = set()
    for gap in gaps:
        gap_info = {
            'type': gap['workflow'],
            'from': gap['date_from'],
            'to': gap.pop('date_to') - timedelta(1) if gap['full_day'] else gap.pop('date_to'),
            'fullDay': gap['full_day'],
            'meta': {
                'work_in_absence': gap['work_in_absence'],
                'id': gap['id'],
                'comment': gap['comment'],
                'show_delete_button': show_delete_button(observer, gap),
                'show_edit_button': show_edit_button(observer, gap),
                'show_external_link_button': gap['workflow'] == 'duty',
                'periodic_gap_id': gap.get('periodic_gap_id'),
            },
        }

        if gap['workflow'] == 'office_work':
            room_id = gap['room']
            office_id = gap['office']
            meta = {
                'place': gap['place'],
                'office': office_id,
                'office_code': None,
                'office_name': None,
                'office_name_en': None,
                'room': room_id,
                'room_name': None,
                'room_name_en': None,
                'room_num': None,
                'table': gap['table'],
            }

            if room_id:
                room_ids.add(room_id)

            if office_id:
                office_ids.add(office_id)

            gap_info['meta'].update(meta)
        if gap['workflow'] == 'duty':
            gap_info['meta'].update(
                {
                    'service_slug': gap['service_slug'],
                    'service_name': gap['service_name'],
                    'shift_id': gap['shift_id'],
                    'role_on_duty': gap['role_on_duty'],
                }
            )
        if gap['workflow'] == 'illness':
            gap_info['meta']['is_covid'] = gap.get('is_covid', False)

        if gap['workflow'] == 'vacation':
            gap_info['meta'].update(
                {
                    'mandatory': gap.get('mandatory', False),
                    'vacation_updated': gap.get('vacation_updated', False),
                    'deadline': gap.get('deadline', None),
                }
            )

        if gap.get('periodic_gap_id'):
            periodic_gap_ids.add(gap.get('periodic_gap_id'))

        res.append(gap_info)

    if office_ids or room_ids or periodic_gap_ids:
        _enrich_gaps(res, office_ids, room_ids, periodic_gap_ids)

    return res
