import re

from staff.lib import waffle

from django.conf import settings
from django.utils.http import urlencode
from django.db.models import Q

from staff.person.models import Staff

import logging
logger = logging.getLogger('person_profile.controllers.search')


self_page_re = re.compile(r'^(me|my)*$')
login_re = re.compile(r'[\w\-\.]+')


def get_person(search_query, fields):
    persons = Staff.objects.values(*fields)

    if waffle.switch_is_active('rkn_mode'):
        # Исключаем уволенных из результата поиска людей,
        # что приводит к редиректу на поиск,
        # где они так же убраны
        persons = persons.filter(Q(is_dismissed=False) | Q(memorial_profile__intranet_status=1))

    if search_query.isdigit():
        return (
            persons
            .filter(work_phone=int(search_query))
            .order_by('is_dismissed')
        )[0]
    elif '@' in search_query:
        return (
            persons
            .filter(work_email__istartswith=search_query)
            .order_by('is_dismissed', 'work_email')
        )[0]
    elif login_re.match(search_query):
        try:
            return persons.get(login=search_query)
        except Staff.DoesNotExist:
            return (
                persons
                .filter(login__istartswith=search_query)
                .order_by('is_dismissed', 'login')
            )[0]
    else:
        raise IndexError


def find_person(search_query, observer, fields):

    has_tilda = search_query.startswith('~')
    if has_tilda:
        search_query = search_query[1:]

    if observer.login == search_query:
        if has_tilda:
            search_query = ''
        else:
            return get_person(observer.login, fields)

    if self_page_re.match(search_query):
        return get_person(observer.login, fields)

    try:
        person = get_person(search_query, fields)
    except IndexError:
        return None

    return person


def get_search_url(search_query):
    url = '{protocol}://{host}/peoplesearch?{query}'.format(
        protocol=settings.SEARCH_PROTOCOL,
        host=settings.SEARCH_HOST,
        query=urlencode({
            'text': search_query,
            'from_service': 'staff',
        }),
    )
    return url
