from itertools import groupby

from django.utils.translation import get_language
from django.conf import settings

from staff.lib.requests import get_ids_repository
from staff.lib.tvm2 import get_tvm_ticket_by_deploy

import logging

logger = logging.getLogger('person_profile.controllers.services')


class ServicesError(Exception):
    pass


ALLOWED_SERVICE_STATES = (
    'develop',
    'supported',
    'needinfo'
)

SERVICE_MEMBERSHIP_STATES = (
    'waiting_deletion',
    'approved',
)


def get_user_service_roles(login):
    roles_data = _format_roles(_fetch_user_roles_from_abc(login))
    sorted_roles_data = sorted(roles_data)  # сортируем по роли
    user_roles = groupby(sorted_roles_data, key=lambda r: r[0])

    for role, services in user_roles:
        services = [
            {'id': service_id, 'name': service_name}
            for role, service_id, service_name in services
        ]
        yield role, services


def _fetch_user_roles_from_abc(login):
    try:
        repository = get_ids_repository(
            service='abc',
            resource_type='service_members',
            user_agent=settings.STAFF_USER_AGENT,
            service_ticket=get_tvm_ticket_by_deploy('abc'),
            api_version=4,
            host=settings.ABC_HOST,
            timeout=settings.ABC_TIMEOUT,
        )

        lookup = {
            'staff': login,
            'service__state__in': ','.join(ALLOWED_SERVICE_STATES),
            'state__in': ','.join(SERVICE_MEMBERSHIP_STATES),
            'service__with__tags__exclude': 'micro',
            'fields': 'role.name,service.name,service.id',
        }

        return repository.getiter(lookup=lookup)

    except Exception as e:
        logger.exception('Can\'t get services from plan API for %s', login)
        raise ServicesError(e)


def _get_translation(data):
    """Дергает из словаря вида {'ru': 'Аркадий', 'en': 'Arkadiy'}
       подходящий под сеттинги сервиса перевод
    """

    primary_language, secondary_language = 'ru', 'en'
    if get_language() != 'ru':
        primary_language, secondary_language = 'en', 'ru'

    result = data[primary_language]
    if not result and result is not False:
        result = data[secondary_language]
    return result


def _format_roles(data):
    for role in data:
        role_name = _get_translation(role['role']['name'])
        service_name = _get_translation(role['service']['name'])
        service_id = role['service']['id']
        yield role_name, service_id, service_name
