import logging

from django.conf import settings
from django.utils import dateparse

from staff.lib import requests


logger = logging.getLogger(__name__)

bot_session = requests.Session()
bot_session.headers['Authorization'] = 'OAuth {token}'.format(token=settings.ROBOT_STAFF_OAUTH_TOKEN)


class Ware(object):
    def __repr__(self):
        return self.__str__()

    @classmethod
    def normalize_name(cls, name):
        if name is None:
            return ''
        words = name.split()
        words = [w.capitalize() if w.isalpha() else w for w in words]
        return ' '.join(words)


class Hardware(Ware):

    def __init__(self, data):
        self.model = '{sub_group} {model}'.format(
            sub_group=self.normalize_name(data.get('sub_group')),
            model=self.normalize_name(data.get('model')),
        ).strip()
        self.group = self.normalize_name(data.get('group'))
        self.code = self.normalize_name(data.get('instance_number'))
        start_date = dateparse.parse_datetime(data.get('start_date', ''))
        self.start_date = start_date.isoformat() if start_date else ''

    def __str__(self):
        return 'Hardware: {group} "{model}" {code}'.format(**vars(self))

    def as_dict(self):
        return {
            'group': self.group,
            'model': self.model,
            'code': self.code,
            'start_date': self.start_date,
        }


class Software(Ware):

    def __init__(self, data):
        self.name = self.normalize_name(data.get('model'))
        self.producer = self.normalize_name(data.get('sub_group'))

    def __str__(self):
        return 'Software: "{name}" {producer}'.format(**vars(self))

    def as_dict(self):
        return {
            'name': self.name,
            'producer': self.producer,
        }


class BotWareLoader(object):

    METHOD = settings.BOT_METHOD
    HOST = settings.BOT_HOST
    TIMEOUT = settings.BOT_WARE_LOADER_TIMEOUT

    def __init__(self, login):
        self.login = login
        self._hardware = None
        self._software = None

    def load(self, ware_class):
        url = '{method}://{host}/api/staff'.format(method=self.METHOD, host=self.HOST)

        try:
            response = bot_session.get('{url}/{ware_type}/?login={login}'.format(
                url=url,
                ware_type=ware_class.__name__.lower(),
                login=self.login,
            ), timeout=self.TIMEOUT)
        except requests.Timeout:
            return
        if response.status_code != 200:
            return

        for ware in response.json():
            yield ware_class(ware)

    @property
    def hardware(self):
        if not self._hardware:
            self._hardware = self.load(Hardware)
        return self._hardware

    def hardware_as_dict(self):
        return [ware.as_dict() for ware in self.hardware]

    @property
    def software(self):
        if not self._software:
            self._software = self.load(Software)
        return self._software

    def software_as_dict(self):
        return [ware.as_dict() for ware in self.software]
