import json
from functools import partial

from django.views.decorators.http import require_http_methods

from staff.lib.decorators import (
    responding_json,
    use_request_lang,
    available_for_external,
)
from staff.lib.forms.staff_form_grid import StaffFormGrid

from ..controllers import contacts as ctl
from ..forms.contacts import ContactForm
from ..views.contacts_view import get_contacts
from staff.lib.forms.errors import (
    invalid_json_error,
    default_error,
)
from ..permissions.utils import observer_has_perm

import logging
logger = logging.getLogger('person_profile.edit_views.edit_contacts_view')

FORM_NAME = 'contacts'


@responding_json
@available_for_external
@require_http_methods(['GET', 'POST'])
@observer_has_perm('edit_contacts')
@use_request_lang
def edit_contacts(request, login):
    grid_cls = partial(StaffFormGrid, form_class=ContactForm)

    if request.method == 'GET':
        form = grid_cls(initial=ctl.ContactsCtl(login).get_person_contacts())
        return {
            'choices': form.choices_as_front_dict(),
            'contacts': form.as_front_dict()
        }

    try:
        contacts_data = json.loads(request.body).get(FORM_NAME, [])
    except (ValueError, AttributeError):
        logger.exception('Wrong JSON: %s', request.body)
        return invalid_json_error(request.body), 400

    form = grid_cls(data=contacts_data)
    if form.is_valid():
        try:
            ctl.ContactsCtl(login).update_contacts(form.cleaned_data)
        except ctl.ContactsError:
            return default_error(), 500
    else:
        return {'errors': {FORM_NAME: form.errors()}}

    return {
        'target': {
            'contacts': get_contacts(login=login, observer_properties=request.permissions_ctl.properties)
        }
    }
