import json

from django.views.decorators.http import require_http_methods

from staff.lib.decorators import (
    responding_json,
    use_request_lang,
    available_for_external,
)
from staff.lib.forms.staff_form_grid import StaffFormGrid

from ..controllers import gpg_key as ctl
from ..forms.gpg_key import GPGKeyForm
from staff.lib.forms.errors import (
    invalid_json_error,
    default_error,
)
from ..permissions.utils import observer_has_perm
from ..views.gpg_keys_view import gpg_keys as gpg_keys_view


import logging
logger = logging.getLogger('person_profile.edit_views.edit_gpg_keys_view')

FORM_NAME = 'gpg_keys'


@responding_json
@available_for_external
@require_http_methods(['GET', 'POST'])
@observer_has_perm('edit_gpg_keys')
@use_request_lang
def edit_gpg_keys(request, login):
    author_login = request.user.username

    if request.method == 'GET':
        try:
            return {
                FORM_NAME: StaffFormGrid(
                    GPGKeyForm, initial=ctl.get_initial(login)
                ).as_front_dict()
            }
        except ctl.GPGKeyError:
            return default_error(), 500

    try:
        form_data = json.loads(request.body).get(FORM_NAME, [])
    except ValueError:
        logger.exception('Wrong JSON: %s', request.body)
        return invalid_json_error(request.body), 400

    grid = StaffFormGrid(GPGKeyForm, data=form_data)
    if not grid.is_valid():
        return {'errors': {FORM_NAME: grid.errors()}}, 400

    try:
        ctl.update(login, grid.cleaned_data, author_login, request.real_user_ip)
    except ctl.GPGKeyError:
        return default_error(), 500

    return gpg_keys_view(request, login)
