import json

from django.views.decorators.http import require_http_methods

from staff.lib.decorators import (
    responding_json,
    use_request_lang,
    available_for_external,
)
from staff.lib.forms.staff_form_grid import StaffFormGrid

from ..controllers import head as ctl
from ..forms.head import HeadForm
from staff.lib.forms.errors import (
    invalid_json_error,
    form_data_error,
    default_error,
)
from ..permissions.utils import observer_has_perm

import logging
logger = logging.getLogger('person_profile.edit_views.edit_head_view')

FORM_NAME = 'head'


@responding_json
@available_for_external
@require_http_methods(['GET', 'POST'])
@observer_has_perm('edit_head')
@use_request_lang
def edit_head(request, login):
    if request.method == 'GET':
        try:
            return {
                FORM_NAME: StaffFormGrid(
                    HeadForm,
                    initial=[ctl.get_initial(login)]
                ).as_front_dict()
            }
        except ctl.HeadError:
            return default_error(), 500

    try:
        form_data = json.loads(request.body)[FORM_NAME][0]
    except ValueError:
        logger.exception('Wrong JSON: %s', request.body)
        return invalid_json_error(request.body), 400
    except (KeyError, IndexError):
        logger.exception('Cannot find form data: %s', FORM_NAME)
        return form_data_error(FORM_NAME), 400

    grid = StaffFormGrid(HeadForm, data=[form_data])
    if not grid.is_valid():
        return {'errors': {FORM_NAME: grid.errors()}}

    try:
        ctl.update(login, grid.cleaned_data[0], request.user)
    except ctl.HeadError:
        return default_error(), 500

    return {'target': {}}
